"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const lib_1 = require("../lib");
const lambda = require("@aws-cdk/aws-lambda");
const dynamodb = require("@aws-cdk/aws-dynamodb");
const cdk = require("@aws-cdk/core");
require("@aws-cdk/assert/jest");
function deployNewFunc(stack) {
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
    };
    return new lib_1.DynamoDBStreamToLambda(stack, 'test-lambda-dynamodb-stack', props);
}
test('snapshot test DynamoDBStreamToLambda default params', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('check lambda EventSourceMapping', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(stack).toHaveResource('AWS::Lambda::EventSourceMapping', {
        EventSourceArn: {
            "Fn::GetAtt": [
                "testlambdadynamodbstackDynamoTable8138E93B",
                "StreamArn"
            ]
        },
        FunctionName: {
            Ref: "testlambdadynamodbstackLambdaFunction5DDB3E8D"
        },
        BatchSize: 100,
        StartingPosition: "TRIM_HORIZON",
        MaximumRecordAgeInSeconds: 86400,
        MaximumRetryAttempts: 500,
        BisectBatchOnFunctionError: true
    });
});
test('check DynamoEventSourceProps override', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        dynamoEventSourceProps: {
            startingPosition: lambda.StartingPosition.LATEST,
            batchSize: 55
        }
    };
    new lib_1.DynamoDBStreamToLambda(stack, 'test-lambda-dynamodb-stack', props);
    expect(stack).toHaveResource('AWS::Lambda::EventSourceMapping', {
        EventSourceArn: {
            "Fn::GetAtt": [
                "testlambdadynamodbstackDynamoTable8138E93B",
                "StreamArn"
            ]
        },
        FunctionName: {
            Ref: "testlambdadynamodbstackLambdaFunction5DDB3E8D"
        },
        BatchSize: 55,
        StartingPosition: "LATEST"
    });
});
test('check lambda permission to read dynamodb stream', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: "dynamodb:ListStreams",
                    Effect: "Allow",
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                {
                                    "Fn::GetAtt": [
                                        "testlambdadynamodbstackDynamoTable8138E93B",
                                        "Arn"
                                    ]
                                },
                                "/stream/*"
                            ]
                        ]
                    }
                },
                {
                    Action: [
                        "dynamodb:DescribeStream",
                        "dynamodb:GetRecords",
                        "dynamodb:GetShardIterator"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testlambdadynamodbstackDynamoTable8138E93B",
                            "StreamArn"
                        ]
                    }
                },
                {
                    Action: [
                        "sqs:SendMessage",
                        "sqs:GetQueueAttributes",
                        "sqs:GetQueueUrl"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testlambdadynamodbstackSqsDlqQueue4CC9868B",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        }
    });
});
test('check dynamodb table stream override', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        dynamoTableProps: {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        }
    };
    new lib_1.DynamoDBStreamToLambda(stack, 'test-lambda-dynamodb-stack', props);
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            }
        ],
        AttributeDefinitions: [
            {
                AttributeName: "id",
                AttributeType: "S"
            }
        ],
        BillingMode: "PAY_PER_REQUEST",
        SSESpecification: {
            SSEEnabled: true
        },
        StreamSpecification: {
            StreamViewType: "NEW_IMAGE"
        }
    });
});
test('check getter methods', () => {
    const stack = new cdk.Stack();
    const construct = deployNewFunc(stack);
    expect(construct.lambdaFunction !== null);
    expect(construct.dynamoTable !== null);
});
test('check exception for Missing existingObj from props', () => {
    const stack = new cdk.Stack();
    const props = {};
    try {
        new lib_1.DynamoDBStreamToLambda(stack, 'test-iot-lambda-integration', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
//# sourceMappingURL=data:application/json;base64,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