//------------------------------------------------------------------------------
// Copyright 2022 H2O.ai
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
//------------------------------------------------------------------------------
// This file is auto-generated by ci/gendoc.py
//------------------------------------------------------------------------------
#include "documentation.h"
namespace dt {


const char* doc_dt_as_type =
R"(as_type(cols, new_type)
--

.. x-version-added:: 1.0

Convert columns `cols` into the prescribed stype.

This function does not modify the data in the original column. Instead
it returns a new column which converts the values into the new type on
the fly.

Parameters
----------
cols: FExpr
    Single or multiple columns that need to be converted.

new_type: Type | stype
    Target type.

return: FExpr
    The output will have the same number of rows and columns as the
    input; column names will be preserved too.


Examples
--------
.. code-block:: python

    >>> from datatable import dt, f, as_type
    >>>
    >>> df = dt.Frame({'A': ['1', '1', '2', '1', '2'],
    ...                'B': [None, '2', '3', '4', '5'],
    ...                'C': [1, 2, 1, 1, 2]})
    >>> df
       | A      B          C
       | str32  str32  int32
    -- + -----  -----  -----
     0 | 1      NA         1
     1 | 1      2          2
     2 | 2      3          1
     3 | 1      4          1
     4 | 2      5          2
    [5 rows x 3 columns]


Convert column A from string to integer type::

    >>> df[:, as_type(f.A, int)]
       |     A
       | int64
    -- + -----
     0 |     1
     1 |     1
     2 |     2
     3 |     1
     4 |     2
    [5 rows x 1 column]


The exact dtype can be specified::

    >>> df[:, as_type(f.A, dt.Type.int32)]
       |     A
       | int32
    -- + -----
     0 |     1
     1 |     1
     2 |     2
     3 |     1
     4 |     2
    [5 rows x 1 column]


Convert multiple columns to different types::

    >>> df[:, [as_type(f.A, int), as_type(f.C, dt.str32)]]
       |     A  C
       | int64  str32
    -- + -----  -----
     0 |     1  1
     1 |     1  2
     2 |     2  1
     3 |     1  1
     4 |     2  2
    [5 rows x 2 columns]
)";


const char* doc_dt_by =
R"(by(*cols, add_columns=True)
--

Group-by clause for use in Frame's square-bracket selector.

Whenever a ``by()`` object is present inside a ``DT[i, j, ...]``
expression, it makes all other expressions to be evaluated in
group-by mode. This mode causes the following changes to the
evaluation semantics:

- A "Groupby" object will be computed for the frame ``DT``, grouping
  it by columns specified as the arguments to the ``by()`` call. This
  object keeps track of which rows of the frame belong to which group.

- If an ``i`` expression is present (row filter), it will be
  interpreted within each group. For example, if ``i`` is a slice,
  then the slice will be applied separately to each group. Similarly,
  if ``i`` expression contains a formula with reduce functions, then
  those functions will be evaluated for each group. For example::

      >>> DT[f.A == max(f.A), :, by(f.group_id)]

  will select those rows where column A reaches its peak value within
  each group (there could be multiple such rows within each group).

- Before ``j`` is evaluated, the ``by()`` clause adds all its columns
  at the start of ``j`` (unless ``add_columns`` argument is ``False``). If
  ``j`` is a "select-all" slice (i.e. ``:``), then those columns will
  also be excluded from the list of all columns so that they will be
  present in the output only once.

- During evaluation of ``j``, the reducer functions, such as
  :func:`min`, :func:`sum`, etc, will be evaluated by-group, that is
  they will find the minimal value in each group, the sum of values in
  each group, and so on. If a reducer expression is combined with a
  regular column expression, then the reduced column will be
  auto-expanded into a column that is constant within each group.

- Note that if both ``i`` and ``j`` contain reducer functions, then
  those functions will have a slightly different notion of groups: the
  reducers in ``i`` will see each group "in full", whereas the
  reducers in ``j`` will see each group after it was filtered by the
  expression in ``i`` (and possibly not even see some of the groups
  at all, if they were filtered out completely).

- If ``j`` contains only reducer expressions, then the final result
  will be a Frame containing just a single row for each
  group. This resulting frame will also be keyed by the grouped-by
  columns.


The ``by()`` function expects a single column or a sequence of columns
as the argument(s). It accepts either a column name, or an
f-expression. In particular, you can perform a group-by on a
dynamically computed expression::

    >>> DT[:, :, by(dt.math.floor(f.A/100))]

The default behavior of groupby is to sort the groups in the ascending
order, with NA values appearing before any other values. As a special
case, if you group by an expression ``-f.A``, then it will be
treated as if you requested to group by the column "A" sorting it in
the descending order. This will work even with column types that are
not arithmetic, for example "A" could be a string column here.


Parameters
----------
cols: FExpr
    Columns to group the frame by.

add_columns: bool
    If ``True``, columns, by which the frame is grouped,
    are added to the beginning of the resulting frame.
    If ``False``, the resulting frame will only contain
    columns listed in ``j``.

return: object
    ``datatable.by`` object for use in square-bracket selector.


Examples
--------
.. code-block:: python

    >>> from datatable import dt, f, by
    >>>
    >>> df = dt.Frame({"group1": ["A", "A", "B", "B", "A"],
    ...                "group2": [1, 0, 1, 1, 1],
    ...                "var1": [343, 345, 567, 345, 212]})
    >>> df
       | group1  group2   var1
       | str32     int8  int32
    -- + ------  ------  -----
     0 | A            1    343
     1 | A            0    345
     2 | B            1    567
     3 | B            1    345
     4 | A            1    212
    [5 rows x 3 columns]


Group by a single column::

    >>> df[:, dt.count(), by("group1")]
       | group1  count
       | str32   int64
    -- + ------  -----
     0 | A           3
     1 | B           2
    [2 rows x 2 columns]


Group by multiple columns::

    >>> df[:, dt.sum(f.var1), by("group1", "group2")]
       | group1  group2   var1
       | str32     int8  int64
    -- + ------  ------  -----
     0 | A            0    345
     1 | A            1    555
     2 | B            1    912
    [3 rows x 3 columns]


Return grouping result without the grouping column(s) by setting the
``add_columns`` parameter to ``False``::

    >>> df[:, dt.sum(f.var1), by("group1", "group2", add_columns=False)]
       |  var1
       | int64
    -- + -----
     0 |   345
     1 |   555
     2 |   912
    [3 rows x 1 column]


:ref:`f-expressions` can be passed to :func:`by()`::

    >>> df[:, dt.count(), by(f.var1 < 400)]
       |    C0  count
       | bool8  int64
    -- + -----  -----
     0 |     0      1
     1 |     1      4
    [2 rows x 2 columns]


By default, the groups are sorted in ascending order. The inverse is
possible by negating the :ref:`f-expressions` in :func:`by()`::

    >>> df[:, dt.count(), by(-f.group1)]
       | group1  count
       | str32   int64
    -- + ------  -----
     0 | B           2
     1 | A           3
    [2 rows x 2 columns]

An integer can be passed to the ``i`` section::

    >>> df[0, :, by("group1")]
       | group1  group2   var1
       | str32     int8  int32
    -- + ------  ------  -----
     0 | A            1    343
     1 | B            1    567
    [2 rows x 3 columns]

A slice is also acceptable within the ``i`` section::

    >>> df[-1:, :, by("group1")]
       | group1  group2   var1
       | str32     int8  int32
    -- + ------  ------  -----
     0 | A            1    212
     1 | B            1    345
    [2 rows x 3 columns]


.. note::

  :ref:`f-expressions` is not implemented yet for the ``i`` section in a
  groupby. Also, a sequence cannot be passed to the ``i`` section in the
  presence of :func:`by()`.


See Also
--------
- :ref:`Grouping with by` user guide for more examples.
)";


const char* doc_dt_categories =
R"(categories(cols)
--

.. x-version-added:: 1.1.0

Get categories for categorical data.

Parameters
----------
cols: FExpr
    Input categorical data.

return: FExpr
    f-expression that returns categories for each column
    from `cols`.

except: TypeError
    The exception is raised when one of the columns from `cols`
    has a non-categorical type.
)";


const char* doc_dt_cbind =
R"(cbind(*frames, force=False)
--

Create a new Frame by appending columns from several `frames`.

This function is exactly equivalent to::

    >>> dt.Frame().cbind(*frames, force=force)

Parameters
----------
frames: Frame | List[Frame] | None
    The list/tuple/sequence/generator expression of Frames to append.
    It may also contain `None` values, which will be simply
    skipped.

force: bool
    If `True`, allows Frames to be appended even if they have unequal
    number of rows. The resulting Frame will have number of rows equal
    to the largest among all Frames. Those Frames which have less
    than the largest number of rows, will be padded with NAs (with the
    exception of Frames having just 1 row, which will be replicated
    instead of filling with NAs).

return: Frame
    A new frame that is created by appending columns from `frames`.


See also
--------
- :func:`rbind()` -- function for row-binding several frames.
- :meth:`dt.Frame.cbind()` -- Frame method for cbinding several frames to
  another.


Examples
--------
.. code-block:: python

    >>> from datatable import dt, f
    >>>
    >>> DT = dt.Frame(A=[1, 2, 3], B=[4, 7, 0])
    >>> DT
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     1      4
     1 |     2      7
     2 |     3      0
    [3 rows x 2 columns]

    >>> frame1 = dt.Frame(N=[-1, -2, -5])
    >>> frame1
       |     N
       | int32
    -- + -----
     0 |    -1
     1 |    -2
     2 |    -5
    [3 rows x 1 column]

    >>> dt.cbind([DT, frame1])
       |     A      B      N
       | int32  int32  int32
    -- + -----  -----  -----
     0 |     1      4     -1
     1 |     2      7     -2
     2 |     3      0     -5
    [3 rows x 3 columns]

If the number of rows are not equal, you can force the binding by setting
the `force` parameter to `True`::

    >>> frame2 = dt.Frame(N=[-1, -2, -5, -20])
    >>> frame2
       |     N
       | int32
    -- + -----
     0 |    -1
     1 |    -2
     2 |    -5
     3 |   -20
    [4 rows x 1 column]

    >>> dt.cbind([DT, frame2], force=True)
       |     A      B      N
       | int32  int32  int32
    -- + -----  -----  -----
     0 |     1      4     -1
     1 |     2      7     -2
     2 |     3      0     -5
     3 |    NA     NA    -20
    [4 rows x 3 columns]
)";


const char* doc_dt_codes =
R"(codes(cols)
--

.. x-version-added:: 1.1.0

Get integer codes for categorical data.

Parameters
----------
cols: FExpr
    Input categorical data.

return: FExpr
    f-expression that returns integer codes for each column
    from `cols`.

except: TypeError
    The exception is raised when one of the columns from `cols`
    has a non-categorical type.
)";


const char* doc_dt_corr =
R"(corr(col1, col2)
--

Calculate the
`Pearson correlation <https://en.wikipedia.org/wiki/Pearson_correlation_coefficient>`_
between `col1` and `col2`.

Parameters
----------
col1, col2: FExpr
    Input columns.

return: Expr
    f-expression having one row, one column and the correlation coefficient
    as the value. If one of the columns is non-numeric, the value is `NA`.
    The column stype is `float32` if both `col1` and `col2` are `float32`,
    and `float64` in all the other cases.


Examples
--------
.. code-block:: python

    >>> from datatable import dt, f
    >>>
    >>> DT = dt.Frame(A = [0, 1, 2, 3], B = [0, 2, 4, 6])
    >>> DT
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     0      0
     1 |     1      2
     2 |     2      4
     3 |     3      6
    [4 rows x 2 columns]

    >>> DT[:, dt.corr(f.A, f.B)]
       |      C0
       | float64
    -- + -------
     0 |       1
    [1 row x 1 column]


See Also
--------
- :func:`cov()` -- function to calculate covariance between two columns.
)";


const char* doc_dt_count =
R"(count(cols)
--

Calculate the number of non-missing values for each column from `cols`.

Parameters
----------
cols: FExpr
    Input columns.

return: Expr
    f-expression having one row, and the same names and number of columns
    as in `cols`. All the returned column stypes are `int64`.

except: TypeError
    The exception is raised when one of the columns from `cols`
    has a non-numeric and non-string type.


Examples
--------

.. code-block:: python

    >>> from datatable import dt, f
    >>>
    >>> df = dt.Frame({'A': [1, 1, 2, 1, 2],
    ...                'B': [None, 2, 3, 4, 5],
    ...                'C': [1, 2, 1, 1, 2]})
    >>> df
       |     A      B      C
       | int32  int32  int32
    -- + -----  -----  -----
     0 |     1     NA      1
     1 |     1      2      2
     2 |     2      3      1
     3 |     1      4      1
     4 |     2      5      2
    [5 rows x 3 columns]

Get the count of all rows::

    >>> df[:, dt.count()]
       | count
       | int32
    -- + -----
     0 |     5
    [1 row x 1 column]

Get the count of column `B` (note how the null row is excluded from the
count result)::

    >>> df[:, dt.count(f.B)]
       |     B
       | int64
    -- + -----
     0 |     4
    [1 row x 1 column]


See Also
--------

- :func:`sum()` -- function to calculate the sum of values.
)";


const char* doc_dt_countna =
R"(countna(cols)
--

.. x-version-added:: 1.1.0

Count the number of NA values for each column from `cols`.

Parameters
----------
cols: FExpr
    Input columns.

return: Expr
    f-expression having one row, and the same names and number of columns
    as in `cols`. All the returned column stypes are `int64`.

except: TypeError
    The exception is raised when one of the columns from `cols`
    has an obj64 type.



Examples
--------

.. code-block:: python

    >>> from datatable import dt, f
    >>>
    >>> df = dt.Frame({'A': [1, 1, 2, None, 1, 2],
    ...                'B': [None, 2, 3, 4, None, 5],
    ...                'C': [1, 2, 1, 1, 2, 4]})
    >>> df
       |     A      B      C
       | int32  int32  int32
    -- + -----  -----  -----
     0 |     1     NA      1
     1 |     1      2      2
     2 |     2      3      1
     3 |    NA      4      1
     4 |     1     NA      2
     5 |     2      5      4
    [6 rows x 3 columns]

Get the count of NAs of all rows:

    >>> df[:, dt.countna(f[:])]
       |     A      B      C
       | int64  int64  int64
    -- + -----  -----  -----
     0 |     1      2      0
    [1 row x 3 columns]

Get the count of NAs of column `B`:

    >>> df[:, dt.countna(f.B)]
       |     B
       | int64
    -- + -----
     0 |     2
    [1 row x 1 column]



See Also
--------

- :func:`count()` -- function to count the number of non-missing values.
)";


const char* doc_dt_cov =
R"(cov(col1, col2)
--

Calculate `covariance <https://en.wikipedia.org/wiki/Covariance>`_
between `col1` and `col2`.


Parameters
----------
col1, col2: FExpr
    Input columns.

return: Expr
    f-expression having one row, one column and the covariance between
    `col1` and `col2` as the value. If one of the input columns is non-numeric,
    the value is `NA`. The output column stype is `float32` if both `col1`
    and `col2` are `float32`, and `float64` in all the other cases.


Examples
--------

.. code-block:: python

    >>> from datatable import dt, f
    >>>
    >>> DT = dt.Frame(A = [0, 1, 2, 3], B = [0, 2, 4, 6])
    >>> DT
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     0      0
     1 |     1      2
     2 |     2      4
     3 |     3      6
    [4 rows x 2 columns]

    >>> DT[:, dt.cov(f.A, f.B)]
       |      C0
       | float64
    -- + -------
     0 | 3.33333
    [1 row x 1 column]


See Also
--------
- :func:`corr()` -- function to calculate correlation between two columns.
)";


const char* doc_dt_cumcount =
R"(cumcount(reverse=False)
--

.. x-version-added:: 1.1.0

Number rows within each group. In the absence of :func:`by()`
the frame is assumed to consist of one group only.


Parameters
----------
reverse: bool
    By default, when this parameter is ``False``, the numbering
    is performed in the ascending order. Otherwise, when
    this parameter is ``True``, the numbering is done
    in the descending order.

return: FExpr
    f-expression that returns row numbers within each group.


Examples
--------

Create a sample datatable frame::

    >>> from datatable import dt, f, by
    >>> DT = dt.Frame(['a', 'a', 'a', 'b', 'b', 'c', 'c', 'c'])
    >>> DT
       | C0
       | str32
    -- + -----
     0 | a
     1 | a
     2 | a
     3 | b
     4 | b
     5 | c
     6 | c
     7 | c
    [8 rows x 1 column]

Number rows within each group in the ascending order::

    >>> DT[:, dt.cumcount(), f.C0]
       | C0        C1
       | str32  int64
    -- + -----  -----
     0 | a          0
     1 | a          1
     2 | a          2
     3 | b          0
     4 | b          1
     5 | c          0
     6 | c          1
     7 | c          2
    [8 rows x 2 columns]

Number rows within each group in the descending order::

    >>> DT[:, dt.cumcount(reverse = True), f.C0]
       | C0        C1
       | str32  int64
    -- + -----  -----
     0 | a          2
     1 | a          1
     2 | a          0
     3 | b          1
     4 | b          0
     5 | c          2
     6 | c          1
     7 | c          0
    [8 rows x 2 columns]

Number rows in the absence of :func:`by()`::

    >>> DT[:, [f.C0, dt.cumcount()]]
       | C0        C1
       | str32  int64
    -- + -----  -----
     0 | a          0
     1 | a          1
     2 | a          2
     3 | b          3
     4 | b          4
     5 | c          5
     6 | c          6
     7 | c          7
    [8 rows x 2 columns]
)";


const char* doc_dt_cummax =
R"(cummax(cols, reverse=False)
--

.. x-version-added:: 1.1.0

For each column from `cols` calculate cumulative maximum. In the presence
of :func:`by()`, the cumulative maximum is computed within each group.

Parameters
----------
cols: FExpr
    Input data for cumulative maximum calculation.

reverse: bool
    If ``False``, computation is done from top to bottom.
    If ``True``, it is done from bottom to top.

return: FExpr
    f-expression that converts input columns into the columns filled
    with the respective cumulative maximums.

except: TypeError
    The exception is raised when one of the columns from `cols`
    has a non-numeric type.


Examples
--------

Create a sample datatable frame::

    >>> from datatable import dt, f, by
    >>> DT = dt.Frame({"A": [2, None, 5, -1, 0],
    ...                "B": [None, None, None, None, None],
    ...                "C": [5.4, 3, 2.2, 4.323, 3], 
    ...                "D": ['a', 'a', 'b', 'b', 'b']})
       |     A     B        C  D    
       | int32  void  float64  str32
    -- + -----  ----  -------  -----
     0 |     2    NA    5.4    a    
     1 |    NA    NA    3      a    
     2 |     5    NA    2.2    b    
     3 |    -1    NA    4.323  b    
     4 |     0    NA    3      b    
    [5 rows x 4 columns]


Calculate the cumulative maximum in a single column::

    >>> DT[:, dt.cummax(f.A)]
       |     A
       | int32
    -- + -----
     0 |     2
     1 |     2
     2 |     5
     3 |     5
     4 |     5
    [5 rows x 1 column]


Calculate the cumulative maximum from bottom to top::

    >>> DT[:, dt.cummax(f.A, reverse=True)]
       |     A
       | int32
    -- + -----
     0 |     5
     1 |     5
     2 |     5
     3 |     0
     4 |     0
    [5 rows x 1 column]


Calculate the cumulative maximum in multiple columns::

    >>> DT[:, dt.cummax(f[:-1])]
       |     A     B        C
       | int32  void  float64
    -- + -----  ----  -------
     0 |     2    NA      5.4
     1 |     2    NA      5.4
     2 |     5    NA      5.4
     3 |     5    NA      5.4
     4 |     5    NA      5.4
    [5 rows x 3 columns]


For a grouped frame calculate the cumulative maximum within each group::

    >>> DT[:, dt.cummax(f[:]), by('D')]
       | D          A     B        C
       | str32  int32  void  float64
    -- + -----  -----  ----  -------
     0 | a          2    NA    5.4  
     1 | a          2    NA    5.4  
     2 | b          5    NA    2.2  
     3 | b          5    NA    4.323
     4 | b          5    NA    4.323
    [5 rows x 4 columns]
)";


const char* doc_dt_cummin =
R"(cummin(cols, reverse=False)
--

.. x-version-added:: 1.1.0

For each column from `cols` calculate cumulative minimum. In the presence of :func:`by()`,
the cumulative minimum is computed within each group.

Parameters
----------
cols: FExpr
    Input data for cumulative minimum calculation.

reverse: bool
    If ``False``, computation is done from top to bottom.
    If ``True``, it is done from bottom to top.

return: FExpr
    f-expression that converts input columns into the columns filled
    with the respective cumulative minimums.

except: TypeError
    The exception is raised when one of the columns from `cols`
    has a non-numeric type.


Examples
--------

Create a sample datatable frame::

    >>> from datatable import dt, f, by
    >>> DT = dt.Frame({"A": [2, None, 5, -1, 0],
    ...                "B": [None, None, None, None, None],
    ...                "C": [5.4, 3, 2.2, 4.323, 3], 
    ...                "D": ['a', 'a', 'b', 'b', 'b']})
       |     A     B        C  D    
       | int32  void  float64  str32
    -- + -----  ----  -------  -----
     0 |     2    NA    5.4    a    
     1 |    NA    NA    3      a    
     2 |     5    NA    2.2    b    
     3 |    -1    NA    4.323  b    
     4 |     0    NA    3      b    
    [5 rows x 4 columns]


Calculate the cumulative minimum in a single column::

    >>> DT[:, dt.cummin(f.A)]
       |     A
       | int32
    -- + -----
     0 |     2
     1 |     2
     2 |     2
     3 |    -1
     4 |    -1
    [5 rows x 1 column]


Calculate the cumulative minimum from bottom to top::

    >>> DT[:, dt.cummin(f.A, reverse=True)]
       |     A
       | int32
    -- + -----
     0 |    -1
     1 |    -1
     2 |    -1
     3 |    -1
     4 |     0
    [5 rows x 1 column]


Calculate the cumulative minimum in multiple columns::

    >>> DT[:, dt.cummin(f[:-1])]
       |     A     B        C
       | int32  void  float64
    -- + -----  ----  -------
     0 |     2    NA      5.4
     1 |     2    NA      3  
     2 |     2    NA      2.2
     3 |    -1    NA      2.2
     4 |    -1    NA      2.2
    [5 rows x 3 columns]


For a grouped frame calculate the cumulative minimum within each group::

    >>> DT[:, dt.cummin(f[:]), by('D')]
       | D          A     B        C
       | str32  int32  void  float64
    -- + -----  -----  ----  -------
     0 | a          2    NA      5.4
     1 | a          2    NA      3  
     2 | b          5    NA      2.2
     3 | b         -1    NA      2.2
     4 | b         -1    NA      2.2
    [5 rows x 4 columns]
)";


const char* doc_dt_cumprod =
R"(cumprod(cols, reverse=False)
--

.. x-version-added:: 1.1.0

For each column from `cols` calculate cumulative product. The product of
the missing values is calculated as one. In the presence of :func:`by()`,
the cumulative product is computed within each group.

Parameters
----------
cols: FExpr
    Input data for cumulative product calculation.

reverse: bool
    If ``False``, computation is done from top to bottom.
    If ``True``, it is done from bottom to top.

return: FExpr
    f-expression that converts input columns into the columns filled
    with the respective cumulative products.

except: TypeError
    The exception is raised when one of the columns from `cols`
    has a non-numeric type.


Examples
--------

Create a sample datatable frame::

    >>> from datatable import dt, f, by
    >>> DT = dt.Frame({"A": [2, None, 5, -1, 0],
    ...                "B": [None, None, None, None, None],
    ...                "C": [5.4, 3, 2.2, 4.323, 3], 
    ...                "D": ['a', 'a', 'b', 'b', 'b']})
       |     A     B        C  D    
       | int32  void  float64  str32
    -- + -----  ----  -------  -----
     0 |     2    NA    5.4    a    
     1 |    NA    NA    3      a    
     2 |     5    NA    2.2    b    
     3 |    -1    NA    4.323  b    
     4 |     0    NA    3      b    
    [5 rows x 4 columns]


Calculate cumulative product in a single column::

    >>> DT[:, dt.cumprod(f.A)]
       |     A
       | int64
    -- + -----
     0 |     2
     1 |     2
     2 |    10
     3 |   -10
     4 |     0
    [5 rows x 1 column]


Calculate the cumulative product from bottom to top::

    >>> DT[:, dt.cumprod(f.A, reverse=True)]
       |     A
       | int64
    -- + -----
     0 |     0
     1 |     0
     2 |     0
     3 |     0
     4 |     0
    [5 rows x 1 column]


Calculate cumulative products in multiple columns::

    >>> DT[:, dt.cumprod(f[:-1])]
       |     A      B        C
       | int64  int64  float64
    -- + -----  -----  -------
     0 |     2      1    5.4  
     1 |     2      1   16.2  
     2 |    10      1   35.64 
     3 |   -10      1  154.072
     4 |     0      1  462.215
    [5 rows x 3 columns]


For a grouped frame calculate cumulative products within each group::

    >>> DT[:, dt.cumprod(f[:]), by('D')]
       | D          A      B        C
       | str32  int64  int64  float64
    -- + -----  -----  -----  -------
     0 | a          2      1   5.4   
     1 | a          2      1  16.2   
     2 | b          5      1   2.2   
     3 | b         -5      1   9.5106
     4 | b          0      1  28.5318
    [5 rows x 4 columns]
)";


const char* doc_dt_cumsum =
R"(cumsum(cols, reverse=False)
--

.. x-version-added:: 1.1.0

For each column from `cols` calculate cumulative sum. The sum of
the missing values is calculated as zero. In the presence of :func:`by()`,
the cumulative sum is computed within each group.

Parameters
----------
cols: FExpr
    Input data for cumulative sum calculation.

reverse: bool
    If ``False``, computation is done from top to bottom.
    If ``True``, it is done from bottom to top.

return: FExpr
    f-expression that converts input columns into the columns filled
    with the respective cumulative sums.

except: TypeError
    The exception is raised when one of the columns from `cols`
    has a non-numeric type.


Examples
--------

Create a sample datatable frame::

    >>> from datatable import dt, f, by
    >>> DT = dt.Frame({"A": [2, None, 5, -1, 0],
    ...                "B": [None, None, None, None, None],
    ...                "C": [5.4, 3, 2.2, 4.323, 3], 
    ...                "D": ['a', 'a', 'b', 'b', 'b']})
       |     A     B        C  D    
       | int32  void  float64  str32
    -- + -----  ----  -------  -----
     0 |     2    NA    5.4    a    
     1 |    NA    NA    3      a    
     2 |     5    NA    2.2    b    
     3 |    -1    NA    4.323  b    
     4 |     0    NA    3      b    
    [5 rows x 4 columns]


Calculate cumulative sum in a single column::

    >>> DT[:, dt.cumsum(f.A)]
       |     A
       | int64
    -- + -----
     0 |     2
     1 |     2
     2 |     7
     3 |     6
     4 |     6
    [5 rows x 1 column]


Calculate the cumulative sum from bottom to top::

    >>> DT[:, dt.cumsum(f.A, reverse=True)]
       |     A
       | int64
    -- + -----
     0 |     6
     1 |     4
     2 |     4
     3 |    -1
     4 |     0
    [5 rows x 1 column]


Calculate cumulative sums in multiple columns::

    >>> DT[:, dt.cumsum(f[:-1])]
       |     A      B        C
       | int64  int64  float64
    -- + -----  -----  -------
     0 |     2      0    5.4  
     1 |     2      0    8.4  
     2 |     7      0   10.6  
     3 |     6      0   14.923
     4 |     6      0   17.923
    [5 rows x 3 columns]


For a grouped frame calculate cumulative sums within each group::

    >>> DT[:, dt.cumsum(f[:]), by('D')]
       | D          A      B        C
       | str32  int64  int64  float64
    -- + -----  -----  -----  -------
     0 | a          2      0    5.4  
     1 | a          2      0    8.4  
     2 | b          5      0    2.2  
     3 | b          4      0    6.523
     4 | b          4      0    9.523
    [5 rows x 4 columns]
)";


const char* doc_dt_cut =
R"(cut(cols, nbins=10, bins=None, right_closed=True)
--

.. x-version-added:: 0.11

For each column from `cols` bin its values into equal-width intervals,
when `nbins` is specified, or into arbitrary-width intervals,
when interval edges are provided as `bins`.

Parameters
----------
cols: FExpr
    Input data for equal-width interval binning.

nbins: int | List[int]
    When a single number is specified, this number of bins
    will be used to bin each column from `cols`.
    When a list or a tuple is provided, each column will be binned
    by using its own number of bins. In the latter case,
    the list/tuple length must be equal to the number of columns
    in `cols`.

bins: List[Frame]
    A list or a tuple of frames for binning the corresponding columns from `cols`.
    Each frame should have only one column that is populated with the edges of
    the binning intervals in strictly increasing order. Number of elements in
    `bins` must be equal to the number of columns in `cols`.

right_closed: bool
    Each binning interval is `half-open`_. This flag indicates whether
    the right edge of the interval is closed, or not.

return: FExpr
    f-expression that converts input columns into the columns filled
    with the respective bin ids.

Examples
--------

Bin one-column frame by specifying a number of bins:

.. code-block:: python

    >>> from datatable import dt, f, cut
    >>> DT = dt.Frame([1, 3, 5])
    >>> DT[:, cut(f[:], nbins = 5)]
       |    C0
       | int32
    -- + -----
     0 |     0
     1 |     2
     2 |     4
    [3 rows x 1 column]

Bin one-column frame by specifying edges of the binning intervals:

.. code-block:: python

    >>> from datatable import dt, f, cut
    >>> DT = dt.Frame([1, 3, 5])
    >>> BINS = [dt.Frame(range(5))]
    >>> DT[:, cut(f[:], bins = BINS)] # Note, "5" goes out of bounds and is binned as "NA"
       |    C0
       | int32
    -- + -----
     0 |     0
     1 |     2
     2 |    NA
    [3 rows x 1 column]


Bin two-column frame by specifying a number of bins for each column separately:

.. code-block:: python

    >>> from datatable import dt, f, cut
    >>> DT = dt.Frame([[1, 3, 5], [5, 7, 9]])
    >>> DT[:, cut(f[:], nbins = [5, 10])]
       |    C0     C1
       | int32  int32
    -- + -----  -----
     0 |     0      0
     1 |     2      4
     2 |     4      9
    [3 rows x 2 columns]


Bin two-column frame by specifying edges of the binning intervals:

.. code-block:: python

    >>> from datatable import dt, f, cut
    >>> DT = dt.Frame([[1, 3, 5], [0.1, -0.1, 1.5]])
    >>> BINS = [dt.Frame(range(10)), dt.Frame([-1.0, 0, 1.0, 2.0])]
    >>> DT[:, cut(f[:], bins = BINS)]
       |    C0     C1
       | int32  int32
    -- + -----  -----
     0 |     0      1
     1 |     2      0
     2 |     4      2
    [3 rows x 2 columns]


See also
--------
:func:`qcut()` -- function for equal-population binning.

.. _`half-open`: https://en.wikipedia.org/wiki/Interval_(mathematics)#Terminology
)";


const char* doc_dt_fillna =
R"(fillna(cols, value=None, reverse=False)
--

.. x-version-added:: 1.1.0

For each column from `cols` either fill the missing values with `value`,
or with the previous/subsequent non-missing values in that column.
In the presence of :func:`by()` the filling is performed group-wise.

Parameters
----------
cols: FExpr
    Input columns.

value: None | bool | int | float | str | list | tuple | dict | FExpr
    A scalar, a list/tuple/dict of values or an f-expression
    to impute missing values with. The number of elements
    in this argument, unless it is a scalar, must match
    the number of columns in the input data.

reverse: bool
    If ``False``, the missing values are filled by using the closest
    previous non-missing values as a replacement. if ``True``,
    the closest subsequent non-missing values are used.

return: FExpr
    f-expression that converts input columns into the columns filled
    with `value`, or with the previous/subsequent non-missing values.

except: ValueError
    The exception is raised when the number of elements in `value`,
    unless it is a scalar, does not match the number of columns in `cols`.


Examples
--------

Create a sample datatable frame::

    >>> from datatable import dt, f, by
    >>> DT = dt.Frame({'building': ['a', 'a', 'b', 'b', 'a', 'a', 'b', 'b'],
    ...                'var1': [1.5, None, 2.1, 2.2, 1.2, 1.3, 2.4, None],
    ...                'var2': [100, 110, 105, None, 102, None, 103, 107],
    ...                'var3': [10, 11, None, None, None, None, None, None],
    ...                'var4': [1, 2, 3, 4, 5, 6, 7, 8]})
       | building     var1   var2   var3   var4
       | str32     float64  int32  int32  int32
    -- + --------  -------  -----  -----  -----
     0 | a             1.5    100     10      1
     1 | a            NA      110     11      2
     2 | b             2.1    105     NA      3
     3 | b             2.2     NA     NA      4
     4 | a             1.2    102     NA      5
     5 | a             1.3     NA     NA      6
     6 | b             2.4    103     NA      7
     7 | b            NA      107     NA      8
    [8 rows x 5 columns]

Fill all the missing values in a column with a single value::

   >>> DT[:, dt.fillna(f.var1, 2)]
      |    var1
      | float64
   -- + -------
    0 |     1.5
    1 |     2  
    2 |     2.1
    3 |     2.2
    4 |     1.2
    5 |     1.3
    6 |     2.4
    7 |     2  
   [8 rows x 1 column]

Fill all the missing values in multiple columns with a single value::

    >>> DT[:, dt.fillna(f[1:], 2)]
       |    var1   var2   var3   var4
       | float64  int32  int32  int32
    -- + -------  -----  -----  -----
     0 |     1.5    100     10      1
     1 |     2      110     11      2
     2 |     2.1    105      2      3
     3 |     2.2      2      2      4
     4 |     1.2    102      2      5
     5 |     1.3      2      2      6
     6 |     2.4    103      2      7
     7 |     2      107      2      8
    [8 rows x 4 columns]

For the grouped frame, fill missing values with the group's mean::

    >>> DT[:, dt.fillna(f[:], dt.mean(f[:])), by('building')]
       | building     var1     var2     var3     var4
       | str32     float64  float64  float64  float64
    -- + --------  -------  -------  -------  -------
     0 | a         1.5          100     10          1
     1 | a         1.33333      110     11          2
     2 | a         1.2          102     10.5        5
     3 | a         1.3          104     10.5        6
     4 | b         2.1          105     NA          3
     5 | b         2.2          105     NA          4
     6 | b         2.4          103     NA          7
     7 | b         2.23333      107     NA          8
    [8 rows x 5 columns]

Fill down on a single column::

    >>> DT[:, dt.fillna(f.var1)]
       |    var1
       | float64
    -- + -------
     0 |     1.5
     1 |     1.5
     2 |     2.1
     3 |     2.2
     4 |     1.2
     5 |     1.3
     6 |     2.4
     7 |     2.4
    [8 rows x 1 column]


Fill up on a single column::

    >>> DT[:, dt.fillna(f.var1, reverse = True)]
       |    var1
       | float64
    -- + -------
     0 |     1.5
     1 |     2.1
     2 |     2.1
     3 |     2.2
     4 |     1.2
     5 |     1.3
     6 |     2.4
     7 |    NA
    [8 rows x 1 column]


Fill down on multiple columns::

     >>> DT[:, dt.fillna(f['var1':])]
        |    var1   var2   var3   var4
        | float64  int32  int32  int32
     -- + -------  -----  -----  -----
      0 |     1.5    100     10      1
      1 |     1.5    110     11      2
      2 |     2.1    105     11      3
      3 |     2.2    105     11      4
      4 |     1.2    102     11      5
      5 |     1.3    102     11      6
      6 |     2.4    103     11      7
      7 |     2.4    107     11      8
     [8 rows x 4 columns]


Fill up on multiple columns::

    >>> DT[:, dt.fillna(f['var1':], reverse = True)]
       |    var1   var2   var3   var4
       | float64  int32  int32  int32
    -- + -------  -----  -----  -----
     0 |     1.5    100     10      1
     1 |     2.1    110     11      2
     2 |     2.1    105     NA      3
     3 |     2.2    102     NA      4
     4 |     1.2    102     NA      5
     5 |     1.3    103     NA      6
     6 |     2.4    103     NA      7
     7 |    NA      107     NA      8
    [8 rows x 4 columns]


Fill down the grouped frame::

    >>> DT[:, dt.fillna(f['var1':]), by('building')]
       | building     var1   var2   var3   var4
       | str32     float64  int32  int32  int32
    -- + --------  -------  -----  -----  -----
     0 | a             1.5    100     10      1
     1 | a             1.5    110     11      2
     2 | a             1.2    102     11      5
     3 | a             1.3    102     11      6
     4 | b             2.1    105     NA      3
     5 | b             2.2    105     NA      4
     6 | b             2.4    103     NA      7
     7 | b             2.4    107     NA      8
    [8 rows x 5 columns]


Fill up the grouped frame::

    >>> DT[:, dt.fillna(f['var1':], reverse = True), by('building')]
       | building     var1   var2   var3   var4
       | str32     float64  int32  int32  int32
    -- + --------  -------  -----  -----  -----
     0 | a             1.5    100     10      1
     1 | a             1.2    110     11      2
     2 | a             1.2    102     NA      5
     3 | a             1.3     NA     NA      6
     4 | b             2.1    105     NA      3
     5 | b             2.2    103     NA      4
     6 | b             2.4    103     NA      7
     7 | b            NA      107     NA      8
    [8 rows x 5 columns]
)";


const char* doc_dt_first =
R"(first(cols)
--

Return the first row for an ``Expr``. If `cols` is an iterable,
simply return the first element.


Parameters
----------
cols: FExpr | iterable
    Input columns or an iterable.

return: Expr | ...
    One-row f-expression that has the same names, stypes and
    number of columns as `cols`. For an iterable the first
    element is returned.


Examples
--------
Function :func:`first()` called on a frame, that is an iterable of columns,
returns the first column::

    >>> from datatable import dt, first, f, by
    >>> DT = dt.Frame({"A": [1, 1, 2, 1, 2],
    ...                "B": [None, 5, 3, 4, 2]})
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     1     NA
     1 |     1      5
     2 |     2      3
     3 |     1      4
     4 |     2      2

    [5 rows x 2 columns]
    >>> first(DT)
       |     A
       | int32
    -- + -----
     0 |     1
     1 |     1
     2 |     2
     3 |     1
     4 |     2

    [5 rows x 1 column]

Called on a set of columns, :func:`first()` returns the first row::

    >>> DT[:, first(f[:])]
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     1     NA

    [1 row x 2 columns]

The same could also be achieved by passing ``0`` to the row selector::

    >>> DT[0, :]
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     1     NA

    [1 row x 2 columns]

To get the first non-missing value in a column, one should additionally employ
a corresponding i-filter::

    >>> DT[f.B != None, first(f.B)]
       |     B
       | int32
    -- + -----
     0 |     5

    [1 row x 1 column]

Function :func:`first()` is group-aware, meaning that it returns the first row
per group in a :func:`by()` context::

    >>> DT[:, first(f.B), by("A")]
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     1     NA
     1 |     2      3

    [2 rows x 2 columns]

To get the first non-missing value per group,
one should first filter out all the missing values from the column in question::

    >>> DT[f.B != None, :][:, first(f.B), by("A")]
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     1      5
     1 |     2      3

    [2 rows x 2 columns]

.. note::

    Filtering out missing values in the row selector will not work in
    a general case, e.g. when one needs to find the first non-missing values
    in several columns.


See Also
--------
- :func:`last()` -- function that returns the last row.
)";


const char* doc_dt_fread =
R"(fread(
--

This function is capable of reading data from a variety of input formats,
producing a :class:`Frame` as the result. The recognized formats are:
CSV, Jay, XLSX, and plain text. In addition, the data may be inside an
archive such as ``.tar``, ``.gz``, ``.zip``, ``.gz2``, and ``.tgz``.


Parameters
----------
anysource: str | bytes | file | Pathlike | List
    The first argument to fread is the input source.
    Multiple types of sources are supported and can be named
    explicitly as `file`, `text`, `cmd`, and `url`. When the source is
    not named, fread will attempt to guess its type. The most common
    type is `file`, but sometimes the argument is resolved as `text`
    (if the string contains newlines) or `url` (if the string starts
    with `https://`, `s3://` or similar).

    Only one argument out of `anysource`, `file`, `text`, `cmd` or
    `url` can be specified at once.

file: str | file | Pathlike
    A file source can be either the name of the file on disk, or a
    python "file-like" object -- i.e. any object having method
    ``.read()``.

    Generally, specifying a file name should be preferred, since
    reading from a Python ``file`` can only be done in a single-threaded
    mode.

    This argument also supports addressing files inside an archive,
    or sheets inside an Excel workbook. Simply write the name of the
    file as if the archive was a folder: `"data.zip/train.csv"`.

text: str | bytes
    Instead of reading data from file, this argument provides the data
    as a simple in-memory blob.

cmd: str
    A command that will be executed in the shell and its output then
    read as text.

url: str
    This parameter can be used to specify the URL of the input file.
    The data will first be downloaded into a temporary directory and
    then read from there. In the end the temporary files will be removed.

    A path to a public S3 bucket is also supported, however, internally
    it first gets converted into the corresponding https URL.

    We use the standard ``urllib.request`` module to download the
    data. Changing the settings of that module, for example installing
    proxy, password, or cookie managers will allow you to customize
    the download process.

columns: ...
    Limit which columns to read from the CSV file.

sep: str | None
    Field separator in the input file. If this value is `None`
    (default) then the separator will be auto-detected. Otherwise it
    must be a single-character string. When ``sep='\n'``, then the
    data will be read in *single-column* mode. Characters
    ``["'`0-9a-zA-Z]`` are not allowed as the separator, as well as
    any non-ASCII characters.

dec: "." | ","
    Decimal point symbol for floating-point numbers.

max_nrows: int
    The maximum number of rows to read from the file. Setting this
    parameter to any negative number is equivalent to have no limit
    at all. Currently this parameter doesn't always work correctly.

header: bool | None
    If `True` then the first line of the CSV file contains the header.
    If `False` then there is no header. By default the presence of the
    header is heuristically determined from the contents of the file.

na_strings: List[str]
    The list of strings that were used in the input file to represent
    NA values.

fill: bool
    If `True` then the lines of the CSV file are allowed to have
    uneven number of fields. All missing fields will be filled with
    NAs in the resulting frame.

encoding: str | None
    If this parameter is provided, then the input will be recoded
    from this encoding into UTF-8 before reading. Any encoding
    registered with the python ``codec`` module can be used.

skip_to_string: str | None
    Start reading the file from the line containing this string. All
    previous lines will be skipped and discarded. This parameter
    cannot be used together with `skip_to_line`.

skip_to_line: int
    If this setting is given, then this many lines in the file will
    be skipped before we start to parse the file. This can be used
    for example when several first lines in the file contain non-CSV
    data and therefore must be skipped. This parameter cannot be
    used together with `skip_to_string`.

skip_blank_lines: bool
    If `True`, then any empty lines in the input will be skipped. If
    this parameter is `False` then: (a) in single-column mode empty
    lines are kept as empty lines; otherwise (b) if `fill=True` then
    empty lines produce a single line filled with NAs in the output;
    otherwise (c) an :exc:`dt.exceptions.IOError` is raised.

strip_whitespace: bool
    If `True`, then the leading/trailing whitespace will be stripped
    from unquoted string fields. Whitespace is always skipped from
    numeric fields.

quotechar: '"' | "'" | "`"
    The character that was used to quote fields in the CSV file. By
    default the double-quote mark `'"'` is assumed.

tempdir: str | None
    Use this directory for storing temporary files as needed. If not
    provided then the system temporary directory will be used, as
    determined via the :ext-mod:`tempfile` Python module.

nthreads: int | None
    Number of threads to use when reading the file. This number cannot
    exceed the number of threads in the pool ``dt.options.nthreads``.
    If `0` or negative number of threads is requested, then it will be
    treated as that many threads less than the maximum. By default
    all threads in the thread pool are used.

verbose: bool
    If `True`, then print detailed information about the internal
    workings of fread to stdout (or to `logger` if provided).

logger: object
    Logger object that will receive verbose information about fread's
    progress. When this parameter is specified, `verbose` mode will
    be turned on automatically.

multiple_sources: "warn" | "error" | "ignore"
    Action that should be taken when the input resolves to multiple
    distinct sources. By default, (`"warn"`) a warning will be issued
    and only the first source will be read and returned as a Frame.
    The `"ignore"` action is similar, except that the extra sources
    will be discarded without a warning. Lastly, an :exc:`dt.exceptions.IOError`
    can be raised if the value of this parameter is `"error"`.

    If you want all sources to be read instead of only the first one
    then consider using :func:`iread()`.

memory_limit: int
    Try not to exceed this amount of memory allocation (in bytes)
    when reading the data. This limit is advisory and not enforced
    very strictly.

    This setting is useful when reading data from a file that is
    substantially larger than the amount of RAM available on your
    machine.

    When this parameter is specified and fread sees that it needs
    more RAM than the limit in order to read the input file, then
    it will dump the data that was read so far into a temporary file
    in binary format. In the end the returned Frame will be partially
    composed from data located on disk, and partially from the data
    in memory. It is advised to either store this data as a Jay file
    or filter and materialize the frame (if not the performance may
    be slow).

return: Frame
    A single :class:`Frame` object is always returned.

    .. x-version-changed:: 0.11.0

        Previously, a ``dict`` of Frames was returned when multiple
        input sources were provided.

except: dt.exceptions.IOError

See Also
--------
- :func:`iread()`
- :ref:`Fread Examples` user guide for usage examples.
)";


const char* doc_dt_ifelse =
R"(ifelse(condition1, value1, condition2, value2, ..., default)
--

.. x-version-added:: 0.11.0

An expression that chooses its value based on one or more
conditions.

This is roughly equivalent to the following Python code::

    >>> result = value1 if condition1 else \
    ...          value2 if condition2 else \
    ...          ...                  else \
    ...          default

For every row this function evaluates the smallest number of expressions
necessary to get the result. Thus, it evaluates `condition1`, `condition2`,
and so on until it finds the first condition that evaluates to `True`.
It then computes and returns the corresponding `value`. If all conditions
evaluate to `False`, then the `default` value is computed and returned.

Also, if any of the conditions produces NA then the result of the expression
also becomes NA without evaluating any further conditions or values.


Parameters
----------
condition1, condition2, ...: FExpr[bool]
    Expressions each producing a single boolean column. These conditions
    will be evaluated in order until we find the one equal to `True`.

value1, value2, ...: FExpr
    Values that will be used when the corresponding condition evaluates
    to `True`. These must be single columns.

default: FExpr
    Value that will be used when all conditions evaluate to `False`.
    This must be a single column.

return: FExpr
    The resulting expression is a single column whose stype is the
    common stype for all `value1`, ..., `default` columns.


Notes
-----
.. x-version-changed:: 1.0.0

    Earlier this function accepted a single condition only.

Examples
--------

Single condition
~~~~~~~~~~~~~~~~
Task: Create a new column `Colour`, where if `Set` is `'Z'` then the
value should be `'Green'`, else `'Red'`::

    >>> from datatable import dt, f, by, ifelse, update
    >>>
    >>> df = dt.Frame("""Type       Set
    ...                   A          Z
    ...                   B          Z
    ...                   B          X
    ...                   C          Y""")
    >>> df[:, update(Colour = ifelse(f.Set == "Z",  # condition
    ...                              "Green",       # if condition is True
    ...                              "Red"))        # if condition is False
    ... ]
    >>> df
       | Type   Set    Colour
       | str32  str32  str32
    -- + -----  -----  ------
     0 | A      Z      Green
     1 | B      Z      Green
     2 | B      X      Red
     3 | C      Y      Red
    [4 rows x 3 columns]


Multiple conditions
~~~~~~~~~~~~~~~~~~~
Task: Create new column ``value`` whose value is taken from columns ``a``,
``b``, or ``c`` -- whichever is nonzero first::

    >>> df = dt.Frame({"a": [0,0,1,2],
    ...                "b": [0,3,4,5],
    ...                "c": [6,7,8,9]})
    >>> df
       |     a      b      c
       | int32  int32  int32
    -- + -----  -----  -----
     0 |     0      0      6
     1 |     0      3      7
     2 |     1      4      8
     3 |     2      5      9
    [4 rows x 3 columns]

    >>> df['value'] = ifelse(f.a > 0, f.a,  # first condition and result
    ...                      f.b > 0, f.b,  # second condition and result
    ...                      f.c)           # default if no condition is True
    >>> df
       |     a      b      c  value
       | int32  int32  int32  int32
    -- + -----  -----  -----  -----
     0 |     0      0      6      6
     1 |     0      3      7      3
     2 |     1      4      8      1
     3 |     2      5      9      2
    [4 rows x 4 columns]
)";


const char* doc_dt_init_styles =
R"(init_styles()
--

Inject datatable's stylesheets into the Jupyter notebook. This
function does nothing when it runs in a normal Python environment
outside of Jupyter.

When datatable runs in a Jupyter notebook, it renders its Frames
as HTML tables. The appearance of these tables is enhanced using
a custom stylesheet, which must be injected into the notebook at
any point on the page. This is exactly what this function does.

Normally, this function is called automatically when datatable
is imported. However, in some circumstances Jupyter erases these
stylesheets (for example, if you run ``import datatable`` cell
twice). In such cases, you may need to call this method manually.
)";


const char* doc_dt_intersect =
R"(intersect(*frames)
--

Find the intersection of sets of values in the `frames`.

Each frame should have only a single column or be empty.
The values in each frame will be treated as a set, and this function will
perform the
`intersection operation <https://en.wikipedia.org/wiki/Intersection_(set_theory)>`_
on these sets, returning those values that are present in each
of the provided `frames`.

Parameters
----------
*frames: Frame, Frame, ...
    Input single-column frames.

return: Frame
    A single-column frame. The column stype is the smallest common
    stype of columns in the `frames`.

except: ValueError | NotImplementedError
    .. list-table::
        :widths: auto
        :class: api-table

        * - :exc:`dt.exceptions.ValueError`
          - raised when one of the input frames has more than one column.

        * - :exc:`dt.exceptions.NotImplementedError`
          - raised when one of the columns has stype `obj64`.


Examples
--------
.. code-block:: python

    >>> from datatable import dt
    >>>
    >>> s1 = dt.Frame([4, 5, 6, 20, 42])
    >>> s2 = dt.Frame([1, 2, 3, 5, 42])
    >>>
    >>> s1
       |    C0
       | int32
    -- + -----
     0 |     4
     1 |     5
     2 |     6
     3 |    20
     4 |    42
    [5 rows x 1 column]

    >>> s2
       |    C0
       | int32
    -- + -----
     0 |     1
     1 |     2
     2 |     3
     3 |     5
     4 |    42
    [5 rows x 1 column]


Intersection of the two frames::

    >>> dt.intersect([s1, s2])
       |    C0
       | int32
    -- + -----
     0 |     5
     1 |    42
    [2 rows x 1 column]



See Also
--------
- :func:`setdiff()` -- calculate the set difference between the frames.
- :func:`symdiff()` -- calculate the symmetric difference between the sets of values in the frames.
- :func:`union()` -- calculate the union of values in the frames.
- :func:`unique()` -- find unique values in a frame.
)";


const char* doc_dt_iread =
R"(iread(
--

This function is similar to :func:`fread()`, but allows reading
multiple sources at once. For example, this can be used when the
input is a list of files, or a glob pattern, or a multi-file archive,
or multi-sheet XLSX file, etc.


Parameters
----------
...: ...
    Most parameters are the same as in :func:`fread()`. All parse
    parameters will be applied to all input files.

errors: "warn" | "raise" | "ignore" | "store"
    What action to take when one of the input sources produces an
    error. Possible actions are: `"warn"` -- each error is converted
    into a warning and emitted to user, the source that produced the
    error is then skipped; `"raise"` -- the errors are raised
    immediately and the iteration stops; `"ignore"` -- the erroneous
    sources are silently ignored; `"store"` -- when an error is
    raised, it is captured and returned to the user, then the iterator
    continues reading the subsequent sources.

(return): Iterator[Frame] | Iterator[Frame|Exception]
    The returned object is an iterator that produces :class:`Frame` s.
    The iterator is lazy: each frame is read only as needed, after the
    previous frame was "consumed" by the user. Thus, the user can
    interrupt the iterator without having to read all the frames.

    Each :class:`Frame` produced by the iterator has a ``.source``
    attribute that describes the source of each frame as best as
    possible. Each source depends on the type of the input: either a
    file name, or a URL, or the name of the file in an archive, etc.

    If the `errors` parameter is `"store"` then the iterator may
    produce either Frames or exception objects.


See Also
--------
- :func:`fread()`
)";


const char* doc_dt_join =
R"(join(frame)
--

Join clause for use in Frame’s square-bracket selector.

This clause is equivalent to the SQL `JOIN`, though for the moment
datatable only supports left outer joins. In order to join,
the `frame` must be :attr:`keyed <dt.Frame.key>` first, and then joined
to another frame `DT` as::

    >>> DT[:, :, join(X)]

provided that `DT` has the column(s) with the same name(s) as
the key in `frame`.

Parameters
----------
frame: Frame
    An input keyed frame to be joined to the current one.

return: Join Object
    In most of the cases the returned object is directly used in the
    Frame’s square-bracket selector.

except: ValueError
    The exception is raised if `frame` is not keyed.

See Also
--------
- :ref:`Tutorial on joins <join tutorial>`

Examples
--------
.. code-block:: python

    >>> df1 = dt.Frame("""    date    X1  X2
    ...                   01-01-2020  H   10
    ...                   01-02-2020  H   30
    ...                   01-03-2020  Y   15
    ...                   01-04-2020  Y   20""")
    >>>
    >>> df2 = dt.Frame("""X1  X3
    ...                   H   5
    ...                   Y   10""")


First, create a key on the right frame (``df2``). Note that the join key
(``X1``) has unique values and has the same name in the left frame (``df1``)::

    >>> df2.key = "X1"

Join is now possible::

    >>> df1[:, :, join(df2)]
       | date        X1        X2     X3
       | str32       str32  int32  int32
    -- + ----------  -----  -----  -----
     0 | 01-01-2020  H         10      5
     1 | 01-02-2020  H         30      5
     2 | 01-03-2020  Y         15     10
     3 | 01-04-2020  Y         20     10
    [4 rows x 4 columns]

You can refer to columns of the joined frame using prefix :data:`g. <dt.g>`, similar to how columns of the left frame can be accessed using prefix :data:`f. <dt.f>`::

    >>> df1[:, update(X2=f.X2 * g.X3), join(df2)]
    >>> df1
       | date        X1        X2
       | str32       str32  int32
    -- + ----------  -----  -----
     0 | 01-01-2020  H         50
     1 | 01-02-2020  H        150
     2 | 01-03-2020  Y        150
     3 | 01-04-2020  Y        200
    [4 rows x 3 columns]
)";


const char* doc_dt_last =
R"(last(cols)
--

Return the last row for an ``Expr``. If `cols` is an iterable,
simply return the last element.


Parameters
----------
cols: FExpr | iterable
    Input columns or an iterable.

return: Expr | ...
    One-row f-expression that has the same names, stypes and
    number of columns as `cols`. For an iterable the last
    element is returned.


Examples
--------
Function :func:`last()` called on a frame, that is an iterable of columns,
returns the last column::

    >>> from datatable import dt, last, f, by
    >>> DT = dt.Frame({"A": [1, 1, 2, 1, 2],
    ...                "B": [None, 5, 3, 4, None]})
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     1     NA
     1 |     1      5
     2 |     2      3
     3 |     1      4
     4 |     2     NA

    [5 rows x 2 columns]
    >>> last(DT)
       |     B
       | int32
    -- + -----
     0 |    NA
     1 |     5
     2 |     3
     3 |     4
     4 |    NA

    [5 rows x 1 column]

Called on a set of columns, :func:`last()` returns the last row::

    >>> DT[:, last(f[:])]
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     2     NA

    [1 row x 2 columns]

The same could also be achieved by passing ``-1`` to the row selector::

    >>> DT[-1, :]
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     2     NA

    [1 row x 2 columns]

To get the last non-missing value in a column, one should additionally employ
a corresponding i-filter::

    >>> DT[f.B != None, last(f.B)]
       |     B
       | int32
    -- + -----
     0 |     4

    [1 row x 1 column]

Function :func:`last()` is group-aware, meaning that it returns the last row
per group in a :func:`by()` context::

    >>> DT[:, last(f.B), by("A")]
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     1      4
     1 |     2     NA

    [2 rows x 2 columns]

To get the last non-missing value per group,
one should first filter out all the missing values from the column in question::

    >>> DT[f.B != None, :][:, last(f.B), by("A")]
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     1      4
     1 |     2      3

    [2 rows x 2 columns]

.. note::

    Filtering out missing values in the row selector will not work in
    a general case, e.g. when one needs to find the last non-missing values
    in several columns.


See Also
--------
- :func:`first()` -- function that returns the first row.
)";


const char* doc_dt_max =
R"(max(cols)
--

Calculate the maximum value for each column from `cols`. It is recommended
to use it as `dt.max()` to prevent conflict with the Python built-in
`max()` function.

Parameters
----------
cols: FExpr
    Input columns.

return: Expr
    f-expression having one row and the same names, stypes and number
    of columns as in `cols`.

except: TypeError
    The exception is raised when one of the columns from `cols`
    has a non-numeric type.


Examples
--------
.. code-block:: python

    >>> from datatable import dt, f, by
    >>>
    >>> df = dt.Frame({'A': [1, 1, 1, 2, 2, 2, 3, 3, 3],
    ...                'B': [3, 2, 20, 1, 6, 2, 3, 22, 1]})
    >>> df
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     1      3
     1 |     1      2
     2 |     1     20
     3 |     2      1
     4 |     2      6
     5 |     2      2
     6 |     3      3
     7 |     3     22
     8 |     3      1
    [9 rows x 2 columns]

Get the maximum from column B::

    >>> df[:, dt.max(f.B)]
       |     B
       | int32
    -- + -----
     0 |    22
    [1 row x 1 column]

Get the maximum of all columns::

    >>> df[:, [dt.max(f.A), dt.max(f.B)]]
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     3     22
    [1 row x 2 columns]

Same as above, but more convenient::

    >>> df[:, dt.max(f[:])]
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     3     22
    [1 row x 2 columns]

You can pass in a dictionary with new column names::

    >>> df[:, dt.max({"A_max": f.A, "B_max": f.B})]
       |   A_max    B_max
       |   int32    int32
    -- + -------  -------
     0 |       3       22
    [1 row x 2 columns]

In the presence of :func:`by()`, it returns the row with the maximum
value per group::

    >>> df[:, dt.max(f.B), by("A")]
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     1     20
     1 |     2      6
     2 |     3     22
    [3 rows x 2 columns]
)";


const char* doc_dt_mean =
R"(mean(cols)
--

Calculate the mean value for each column from `cols`.


Parameters
----------
cols: FExpr
    Input columns.

return: Expr
    f-expression having one row, and the same names and number of columns
    as in `cols`. The column stypes are `float32` for
    `float32` columns, and `float64` for all the other numeric types.

except: TypeError
    The exception is raised when one of the columns from `cols`
    has a non-numeric type.


See Also
--------

- :func:`median()` -- function to calculate median values.
- :func:`sd()` -- function to calculate standard deviation.


Examples
--------
.. code-block:: python

    >>> from datatable import dt, f, by
    >>>
    >>> df = dt.Frame({'A': [1, 1, 2, 1, 2],
    ...                'B': [None, 2, 3, 4, 5],
    ...                'C': [1, 2, 1, 1, 2]})
    >>>
    >>> df
       |     A      B      C
       | int32  int32  int32
    -- + -----  -----  -----
     0 |     1     NA      1
     1 |     1      2      2
     2 |     2      3      1
     3 |     1      4      1
     4 |     2      5      2
    [5 rows x 3 columns]


Get the mean from column A::

    >>> df[:, dt.mean(f.A)]
       |       A
       | float64
    -- + -------
     0 |     1.4
    [1 row x 1 column]

Get the mean of multiple columns::

    >>> df[:, dt.mean([f.A, f.B])]
       |       A        B
       | float64  float64
    -- + -------  -------
     0 |     1.4      3.5
    [1 row x 2 columns]

Same as above, but applying to a column slice::

    >>> df[:, dt.mean(f[:2])]
       |       A        B
       | float64  float64
    -- + -------  -------
     0 |     1.4      3.5
    [1 row x 2 columns]

You can pass in a dictionary with new column names::

    >>> df[:, dt.mean({"A_mean": f.A, "C_avg": f.C})]
       |  A_mean    C_avg
       | float64  float64
    -- + -------  -------
     0 |     1.4      1.4
    [1 row x 2 columns]

In the presence of :func:`by()`, it returns the average of each column per group::

    >>> df[:, dt.mean({"A_mean": f.A, "B_mean": f.B}), by("C")]
       |     C   A_mean   B_mean
       | int32  float64  float64
    -- + -----  -------  -------
     0 |     1  1.33333      3.5
     1 |     2  1.5          3.5
    [2 rows x 3 columns]
)";


const char* doc_dt_median =
R"(median(cols)
--

Calculate the median value for each column from `cols`.


Parameters
----------
cols: FExpr
    Input columns.

return: Expr
    f-expression having one row, and the same names, stypes and
    number of columns as in `cols`.

except: TypeError
    The exception is raised when one of the columns from `cols`
    has a non-numeric type.


Examples
--------
.. code-block:: python

    >>> from datatable import dt, f, by
    >>>
    >>> df = dt.Frame({'A': [1, 1, 2, 1, 2],
    ...                'B': [None, 2, 3, 4, 5],
    ...                'C': [1, 2, 1, 1, 2]})
    >>>
    >>> df
       |     A      B      C
       | int32  int32  int32
    -- + -----  -----  -----
     0 |     1     NA      1
     1 |     1      2      2
     2 |     2      3      1
     3 |     1      4      1
     4 |     2      5      2
    [5 rows x 3 columns]

Get the median from column A::

    >>> df[:, dt.median(f.A)]
       |       A
       | float64
    -- + -------
     0 |       1
    [1 row x 1 column]

Get the median of multiple columns::

    >>> df[:, dt.median([f.A, f.B])]
       |       A        B
       | float64  float64
    -- + -------  -------
     0 |       1      3.5
    [1 row x 2 columns]

Same as above, but more convenient::

    >>> df[:, dt.median(f[:2])]
       |       A        B
       | float64  float64
    -- + -------  -------
     0 |       1      3.5
    [1 row x 2 columns]

You can pass in a dictionary with new column names::

    >>> df[:, dt.median({"A_median": f.A, "C_mid": f.C})]
       | A_median    C_mid
       |  float64  float64
    -- + --------  -------
     0 |        1        1
    [1 row x 2 columns]

In the presence of :func:`by()`, it returns the median of each column
per group::

    >>> df[:, dt.median({"A_median": f.A, "B_median": f.B}), by("C")]
       |     C  A_median  B_median
       | int32   float64   float64
    -- + -----  --------  --------
     0 |     1       1         3.5
     1 |     2       1.5       3.5
    [2 rows x 3 columns]


See Also
--------

- :func:`mean()` -- function to calculate mean values.
- :func:`sd()` -- function to calculate standard deviation.
)";


const char* doc_dt_min =
R"(min(cols)
--

Calculate the minimum value for each column from `cols`. It is recommended
to use it as `dt.min()` to prevent conflict with the Python built-in
`min()` function.


Parameters
----------
cols: FExpr
    Input columns.

return: Expr
    f-expression having one row and the same names, stypes and number
    of columns as in `cols`.

except: TypeError
    The exception is raised when one of the columns from `cols`
    has a non-numeric type.


Examples
--------
.. code-block:: python

    >>> from datatable import dt, f, by
    >>>
    >>> df = dt.Frame({'A': [1, 1, 1, 2, 2, 2, 3, 3, 3],
    ...                'B': [3, 2, 20, 1, 6, 2, 3, 22, 1]})
    >>>
    >>> df
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     1      3
     1 |     1      2
     2 |     1     20
     3 |     2      1
     4 |     2      6
     5 |     2      2
     6 |     3      3
     7 |     3     22
     8 |     3      1
    [9 rows x 2 columns]

Get the minimum from column B::

    >>> df[:, dt.min(f.B)]
       |     B
       | int32
    -- + -----
     0 |     1
    [1 row x 1 column]

Get the minimum of all columns::

    >>> df[:, [dt.min(f.A), dt.min(f.B)]]
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     1      1
    [1 row x 2 columns]

Same as above, but using the slice notation::

    >>> df[:, dt.min(f[:])]
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     1      1
    [1 row x 2 columns]

You can pass in a dictionary with new column names::

    >>> df[:, dt.min({"A_min": f.A, "B_min": f.B})]
       |   A_min    B_min
       |   int32    int32
    -- + -------  -------
     0 |       1        1
    [1 row x 2 columns]

In the presence of :func:`by()`, it returns the row with the minimum value
per group::

    >>> df[:, dt.min(f.B), by("A")]
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     1      2
     1 |     2      1
     2 |     3      1
    [3 rows x 2 columns]
)";


const char* doc_dt_ngroup =
R"(ngroup(reverse=False)
--

.. x-version-added:: 1.1.0

For each row return a group number it belongs to. In the absence of
:func:`by()` the frame is assumed to consist of one group only.


Parameters
----------
reverse: bool
    By default, when this parameter is ``False``, groups
    are numbered in the ascending order. Otherwise, when
    this parameter is ``True``, groups are numbered
    in the descending order.

return: FExpr
    f-expression that returns group numbers for each row.


Examples
--------

Create a sample datatable frame::

    >>> from datatable import dt, f, by
    >>> DT = dt.Frame(['a', 'a', 'a', 'b', 'b', 'c', 'c', 'c'])
    >>> DT
       | C0
       | str32
    -- + -----
     0 | a
     1 | a
     2 | a
     3 | b
     4 | b
     5 | c
     6 | c
     7 | c
    [8 rows x 1 column]

Number groups in the ascending order::

    >>> DT[:, dt.ngroup(), f.C0]
       | C0        C1
       | str32  int64
    -- + -----  -----
     0 | a          0
     1 | a          0
     2 | a          0
     3 | b          1
     4 | b          1
     5 | c          2
     6 | c          2
     7 | c          2
    [8 rows x 2 columns]

Number groups in the descending order::

    >>> DT[:, dt.ngroup(reverse = True), f.C0]
       | C0        C1
       | str32  int64
    -- + -----  -----
     0 | a          2
     1 | a          2
     2 | a          2
     3 | b          1
     4 | b          1
     5 | c          0
     6 | c          0
     7 | c          0
    [8 rows x 2 columns]

Number groups in the absence of :func:`by()`::

    >>> DT[:, dt.ngroup()]
       | C0        C1
       | str32  int64
    -- + -----  -----
     0 | a          0
     1 | a          0
     2 | a          0
     3 | b          0
     4 | b          0
     5 | c          0
     6 | c          0
     7 | c          0
    [8 rows x 2 columns]
)";


const char* doc_dt_nunique =
R"(nunique(cols)
--

.. x-version-added:: 1.1.0

Count the number of unique values for each column from `cols`.

Parameters
----------
cols: FExpr
    Input columns.

return: Expr
    f-expression having one row, and the same names and number of columns
    as in `cols`. All the returned column stypes are `int64`.

except: TypeError
    The exception is raised when one of the columns from `cols`
    has an unsupported type, e.g. `obj64`.



Examples
--------

.. code-block:: python

    >>> from datatable import dt, f
    >>>
    >>> df = dt.Frame({'A': [1, 1, 2, None, 1, 2],
    ...                'B': [None, 2, 3, 4, None, 5],
    ...                'C': [1, 2, 1, 1, 2, 4]})
    >>> df
       |     A      B      C
       | int32  int32  int32
    -- + -----  -----  -----
     0 |     1     NA      1
     1 |     1      2      2
     2 |     2      3      1
     3 |    NA      4      1
     4 |     1     NA      2
     5 |     2      5      4
    [6 rows x 3 columns]

Count the number of unique values for each column in the frame:

    >>> df[:, dt.nunique(f[:])]
       |     A      B      C
       | int64  int64  int64
    -- + -----  -----  -----
     0 |     2      4      3
    [1 row x 3 columns]

Count the number of unique values for column `B` only:

    >>> df[:, dt.nunique(f.B)]
       |     B
       | int64
    -- + -----
     0 |     4
    [1 row x 1 column]
)";


const char* doc_dt_qcut =
R"(qcut(cols, nquantiles=10)
--

.. x-version-added:: 0.11

Bin all the columns from `cols` into intervals with approximately
equal populations. Thus, the intervals are chosen according to
the sample quantiles of the data.

If there are duplicate values in the data, they will all be placed
into the same bin. In extreme cases this may cause the bins to be
highly unbalanced.

Parameters
----------
cols: FExpr
    Input data for quantile binning.

nquantiles: int | List[int]
    When a single number is specified, this number of quantiles
    will be used to bin each column from `cols`.

    When a list or a tuple is provided, each column will be binned
    by using its own number of quantiles. In the latter case,
    the list/tuple length must be equal to the number of columns
    in `cols`.

return: FExpr
    f-expression that converts input columns into the columns filled
    with the respective quantile ids.


Examples
--------

Bin two-column frame by using the same number of quantiles for both columns:

.. code-block:: python

    >>> from datatable import dt, f, qcut
    >>> DT = dt.Frame([range(5), [3, 14, 15, 92, 6]])
    >>> DT[:, qcut(f[:], nquantiles = 3)]
       |    C0     C1
       | int32  int32
    -- + -----  -----
     0 |     0      0
     1 |     0      1
     2 |     1      2
     3 |     2      2
     4 |     2      0
    [5 rows x 2 columns]


Bin two-column frame by using column-specific number of quantiles:

.. code-block:: python

    >>> from datatable import dt, f, qcut
    >>> DT = dt.Frame([range(5), [3, 14, 15, 92, 6]])
    >>> DT[:, qcut(f[:], nquantiles = [3, 5])]
       |    C0     C1
       | int32  int32
    -- + -----  -----
     0 |     0      0
     1 |     0      2
     2 |     1      3
     3 |     2      4
     4 |     2      1
    [5 rows x 2 columns]


See also
--------
:func:`cut()` -- function for equal-width interval binning.
)";


const char* doc_dt_rbind =
R"(rbind(*frames, force=False, bynames=True)
--

Produce a new frame by appending rows of several `frames`.

This function is equivalent to::

    >>> dt.Frame().rbind(*frames, force=force, by_names=by_names)


Parameters
----------
frames: Frame | List[Frame] | None

force: bool
    If `True`, then the frames are allowed to have mismatching set of
    columns (either different counts or different names). Any gaps in
    the data will be filled with NAs.

    In addition, when this parameter is `True`, rbind will no longer
    produce an error when combining columns of unrelated types.
    Instead, both columns will be converted into strings.

bynames: bool
    Whether to match column names when rbinding.

return: Frame
    A new frame that is created by appending rows from `frames`.


See also
--------
- :func:`cbind()` -- function for col-binding several frames.
- :meth:`dt.Frame.rbind()` -- Frame method for rbinding some frames to
  another.


Examples
--------
.. code-block:: python

    >>> from datatable import dt
    >>>
    >>> DT1 = dt.Frame({"Weight": [5, 4, 6], "Height": [170, 172, 180]})
    >>> DT1
       | Weight  Height
       |  int32   int32
    -- + ------  ------
     0 |      5     170
     1 |      4     172
     2 |      6     180
    [3 rows x 2 columns]

    >>> DT2 = dt.Frame({"Height": [180, 181, 169], "Weight": [4, 4, 5]})
    >>> DT2
       | Weight  Height
       |  int32   int32
    -- + ------  ------
     0 |      4     180
     1 |      4     181
     2 |      5     169
    [3 rows x 2 columns]

    >>> dt.rbind(DT1, DT2)
       | Weight  Height
       |  int32   int32
    -- + ------  ------
     0 |      5     170
     1 |      4     172
     2 |      6     180
     3 |      4     180
     4 |      4     181
     5 |      5     169
    [6 rows x 2 columns]

:func:`rbind()` by default combines frames by names. The frames can also be
bound by column position by setting the `bynames` parameter to ``False``::

    >>> dt.rbind(DT1, DT2, bynames = False)
       | Weight  Height
       |  int32   int32
    -- + ------  ------
     0 |      5     170
     1 |      4     172
     2 |      6     180
     3 |    180       4
     4 |    181       4
     5 |    169       5
    [6 rows x 2 columns]


If the number of columns are not equal or the column names are different,
you can force the row binding by setting the `force` parameter to `True`::

    >>> DT2["Age"] = dt.Frame([25, 50, 67])
    >>> DT2
       | Weight  Height    Age
       |  int32   int32  int32
    -- + ------  ------  -----
     0 |      4     180     25
     1 |      4     181     50
     2 |      5     169     67
    [3 rows x 3 columns]

    >>> dt.rbind(DT1, DT2, force = True)
       | Weight  Height    Age
       |  int32   int32  int32
    -- + ------  ------  -----
     0 |      5     170     NA
     1 |      4     172     NA
     2 |      6     180     NA
     3 |      4     180     25
     4 |      4     181     50
     5 |      5     169     67
    [6 rows x 3 columns]
)";


const char* doc_dt_repeat =
R"(repeat(frame, n)
--

Concatenate ``n`` copies of the ``frame`` by rows and return the result.

This is equivalent to ``dt.rbind([frame] * n)``.


Example
-------
::

    >>> from datatable import dt
    >>> DT = dt.Frame({"A": [1, 1, 2, 1, 2],
    ...                "B": [None, 2, 3, 4, 5]})
    >>> DT
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     1     NA
     1 |     1      2
     2 |     2      3
     3 |     1      4
     4 |     2      5
    [5 rows x 2 columns]

::

    >>> dt.repeat(DT, 2)
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     1     NA
     1 |     1      2
     2 |     2      3
     3 |     1      4
     4 |     2      5
     5 |     1     NA
     6 |     1      2
     7 |     2      3
     8 |     1      4
     9 |     2      5
    [10 rows x 2 columns]
)";


const char* doc_dt_rowall =
R"(rowall(*cols)
--

For each row in `cols` return `True` if all values in that row are `True`,
or otherwise return `False`. The function uses shortcut evaluation:
if the `False` value is found in one of the columns, then the subsequent columns
are skipped. Missing values are counted as `False`.

Parameters
----------
cols: FExpr[bool]
    Input boolean columns.

return: FExpr[bool]
    f-expression consisting of one boolean column that has the same number
    of rows as in `cols`.

except: TypeError
    The exception is raised when one of the columns from `cols`
    has a non-boolean type.


Examples
--------
::

    >>> from datatable import dt, f
    >>> DT = dt.Frame({"A": [True, True],
    ...                "B": [True, False],
    ...                "C": [True, True]})
    >>> DT
       |     A      B      C
       | bool8  bool8  bool8
    -- + -----  -----  -----
     0 |     1      1      1
     1 |     1      0      1
    [2 rows x 3 columns]

::

    >>> DT[:, dt.rowall(f[:])]
       |    C0
       | bool8
    -- + -----
     0 |     1
     1 |     0
    [2 rows x 1 column]


See Also
--------
- :func:`rowany()` -- row-wise `any() <https://docs.python.org/3/library/functions.html#any>`_ function.
)";


const char* doc_dt_rowany =
R"(rowany(*cols)
--

For each row in `cols` return `True` if any of the values in that row
are `True`, or otherwise return `False`. The function uses shortcut evaluation:
if the `True` value is found in one of the columns, then the subsequent columns
are skipped. Missing values are counted as `False`.


Parameters
----------
cols: FExpr[bool]
    Input boolean columns.

return: FExpr[bool]
    f-expression consisting of one boolean column that has the same number
    of rows as in `cols`.

except: TypeError
    The exception is raised when one of the columns from `cols`
    has a non-boolean type.


Examples
--------
::

    >>> from datatable import dt, f
    >>> DT = dt.Frame({"A":[True, True],
    ...                "B":[True, False],
    ...                "C":[True, True]})
    >>> DT
       |     A      B      C
       | bool8  bool8  bool8
    -- + -----  -----  -----
     0 |     1      1      1
     1 |     1      0      1
    [2 rows x 3 columns]

::

    >>> DT[:, dt.rowany(f[:])]
       |    C0
       | bool8
    -- + -----
     0 |     1
     1 |     1
    [2 rows x 1 column]


See Also
--------
- :func:`rowall()` -- row-wise `all() <https://docs.python.org/3/library/functions.html#all>`_ function.
)";


const char* doc_dt_rowargmax =
R"(rowargmax(*cols)
--

.. x-version-added:: 1.1.0

For each row, find the index of the largest value among the columns from `cols`.
When the largest value occurs more than once, the smallest column index
is returned.


Parameters
----------
cols: FExpr
    Input columns.

return: FExpr
    f-expression consisting of one column that has the same number of rows
    as in `cols`. The column type is `int64`.

except: TypeError
    The exception is raised when `cols` has non-numeric columns.


Examples
--------
::

    >>> from datatable import dt, f
    >>> DT = dt.Frame({"A": [1, 1, 2, 1, 2],
    ...                "B": [None, 2, 3, 4, None],
    ...                "C":[True, False, False, True, True]})
    >>> DT
       |     A      B      C
       | int32  int32  bool8
    -- + -----  -----  -----
     0 |     1     NA      1
     1 |     1      2      0
     2 |     2      3      0
     3 |     1      4      1
     4 |     2     NA      1
    [5 rows x 3 columns]

::

    >>> DT[:, dt.rowargmax(f[:])]
       |    C0
       | int64
    -- + -----
     0 |     0
     1 |     1
     2 |     1
     3 |     1
     4 |     0
    [5 rows x 1 column]


See Also
--------
- :func:`rowargmin()` -- find the index of the smallest element row-wise.
)";


const char* doc_dt_rowargmin =
R"(rowargmin(*cols)
--

.. x-version-added:: 1.1.0

For each row, find the smallest value among the columns from `cols`.


Parameters
----------
cols: FExpr
    Input columns.

return: FExpr
    f-expression consisting of one column that has the same number of rows
    as in `cols`. The column stype `int64`.

except: TypeError
    The exception is raised when `cols` has non-numeric columns.


Examples
--------
::

    >>> from datatable import dt, f
    >>> DT = dt.Frame({"A": [1, 1, 2, 1, 2],
    ...                "B": [None, 2, 3, 4, None],
    ...                "C":[True, False, False, True, True]})
    >>> DT
       |     A      B      C
       | int32  int32  bool8
    -- + -----  -----  -----
     0 |     1     NA      1
     1 |     1      2      0
     2 |     2      3      0
     3 |     1      4      1
     4 |     2     NA      1
    [5 rows x 3 columns]

::

    >>> DT[:, dt.rowargmin(f[:])]
       |    C0
       | int64
    -- + -----
     0 |     0
     1 |     2
     2 |     2
     3 |     0
     4 |     2
    [5 rows x 1 column]


See Also
--------
- :func:`rowargmax()` -- find the index of the largest element row-wise.
)";


const char* doc_dt_rowcount =
R"(rowcount(*cols)
--

For each row, count the number of non-missing values in `cols`.


Parameters
----------
cols: FExpr
    Input columns.

return: FExpr
    f-expression consisting of one `int32` column and the same number
    of rows as in `cols`.


Examples
--------
::

    >>> from datatable import dt, f
    >>> DT = dt.Frame({"A": [1, 1, 2, 1, 2],
    ...                "B": [None, 2, 3, 4, None],
    ...                "C":[True, False, False, True, True]})
    >>> DT
       |     A      B      C
       | int32  int32  bool8
    -- + -----  -----  -----
     0 |     1     NA      1
     1 |     1      2      0
     2 |     2      3      0
     3 |     1      4      1
     4 |     2     NA      1
    [5 rows x 3 columns]



Note the exclusion of null values in the count::

    >>> DT[:, dt.rowcount(f[:])]
       |    C0
       | int32
    -- + -----
     0 |     2
     1 |     3
     2 |     3
     3 |     3
     4 |     2
    [5 rows x 1 column]


See Also
--------
- :func:`rowsum()` -- sum of all values row-wise.
)";


const char* doc_dt_rowfirst =
R"(rowfirst(*cols)
--

For each row, find the first non-missing value in `cols`. If all values
in a row are missing, then this function will also produce a missing value.


Parameters
----------
cols: FExpr
    Input columns.

return: FExpr
    f-expression consisting of one column and the same number
    of rows as in `cols`.

except: TypeError
    The exception is raised when input columns have incompatible types.


Examples
--------
::

    >>> from datatable import dt, f
    >>> DT = dt.Frame({"A": [1, 1, 2, 1, 2],
    ...                "B": [None, 2, 3, 4, None],
    ...                "C": [True, False, False, True, True]})
    >>> DT
       |     A      B      C
       | int32  int32  bool8
    -- + -----  -----  -----
     0 |     1     NA      1
     1 |     1      2      0
     2 |     2      3      0
     3 |     1      4      1
     4 |     2     NA      1
    [5 rows x 3 columns]

::

    >>> DT[:, dt.rowfirst(f[:])]
       |    C0
       | int32
    -- + -----
     0 |     1
     1 |     1
     2 |     2
     3 |     1
     4 |     2
    [5 rows x 1 column]

::

    >>> DT[:, dt.rowfirst(f['B', 'C'])]
       |    C0
       | int32
    -- + -----
     0 |     1
     1 |     2
     2 |     3
     3 |     4
     4 |     1
    [5 rows x 1 column]


See Also
--------
- :func:`rowlast()` -- find the last non-missing value row-wise.
)";


const char* doc_dt_rowlast =
R"(rowlast(*cols)
--

For each row, find the last non-missing value in `cols`. If all values
in a row are missing, then this function will also produce a missing value.


Parameters
----------
cols: FExpr
    Input columns.

return: FExpr
    f-expression consisting of one column and the same number
    of rows as in `cols`.

except: TypeError
    The exception is raised when input columns have incompatible types.


Examples
--------
::

    >>> from datatable import dt, f
    >>> DT = dt.Frame({"A": [1, 1, 2, 1, 2],
    ...                "B": [None, 2, 3, 4, None],
    ...                "C":[True, False, False, True, True]})
    >>> DT
       |     A      B      C
       | int32  int32  bool8
    -- + -----  -----  -----
     0 |     1     NA      1
     1 |     1      2      0
     2 |     2      3      0
     3 |     1      4      1
     4 |     2     NA      1
    [5 rows x 3 columns]

::

    >>> DT[:, dt.rowlast(f[:])]
       |    C0
       | int32
    -- + -----
     0 |     1
     1 |     0
     2 |     0
     3 |     1
     4 |     1
    [5 rows x 1 column]

::

    >>> DT[[1, 3], 'C'] = None
    >>> DT
       |     A      B      C
       | int32  int32  bool8
    -- + -----  -----  -----
     0 |     1     NA      1
     1 |     1      2     NA
     2 |     2      3      0
     3 |     1      4     NA
     4 |     2     NA      1
    [5 rows x 3 columns]

::

    >>> DT[:, dt.rowlast(f[:])]
       |    C0
       | int32
    -- + -----
     0 |     1
     1 |     2
     2 |     0
     3 |     4
     4 |     1
    [5 rows x 1 column]


See Also
--------
- :func:`rowfirst()` -- find the first non-missing value row-wise.
)";


const char* doc_dt_rowmax =
R"(rowmax(*cols)
--

For each row, find the largest value among the columns from `cols`.


Parameters
----------
cols: FExpr
    Input columns.

return: FExpr
    f-expression consisting of one column that has the same number of rows
    as in `cols`. The column stype is the smallest common stype
    for `cols`, but not less than `int32`.

except: TypeError
    The exception is raised when `cols` has non-numeric columns.


Examples
--------
::

    >>> from datatable import dt, f
    >>> DT = dt.Frame({"A": [1, 1, 2, 1, 2],
    ...                "B": [None, 2, 3, 4, None],
    ...                "C":[True, False, False, True, True]})
    >>> DT
       |     A      B      C
       | int32  int32  bool8
    -- + -----  -----  -----
     0 |     1     NA      1
     1 |     1      2      0
     2 |     2      3      0
     3 |     1      4      1
     4 |     2     NA      1
    [5 rows x 3 columns]

::

    >>> DT[:, dt.rowmax(f[:])]
       |    C0
       | int32
    -- + -----
     0 |     1
     1 |     2
     2 |     3
     3 |     4
     4 |     2
    [5 rows x 1 column]


See Also
--------
- :func:`rowmin()` -- find the smallest element row-wise.
)";


const char* doc_dt_rowmean =
R"(rowmean(*cols)
--

For each row, find the mean values among the columns from `cols`
skipping missing values. If a row contains only the missing values,
this function will produce a missing value too.


Parameters
----------
cols: FExpr
    Input columns.

return: FExpr
    f-expression consisting of one column that has the same number of rows
    as in `cols`. The column stype is `float32` when all the `cols`
    are `float32`, and `float64` in all the other cases.

except: TypeError
    The exception is raised when `cols` has non-numeric columns.

Examples
--------
::

    >>> from datatable import dt, f, rowmean
    >>> DT = dt.Frame({'a': [None, True, True, True],
    ...                'b': [2, 2, 1, 0],
    ...                'c': [3, 3, 1, 0],
    ...                'd': [0, 4, 6, 0],
    ...                'q': [5, 5, 1, 0]}

    >>> DT
       |     a      b      c      d      q
       | bool8  int32  int32  int32  int32
    -- + -----  -----  -----  -----  -----
     0 |    NA      2      3      0      5
     1 |     1      2      3      4      5
     2 |     1      1      1      6      1
     3 |     1      0      0      0      0
    [4 rows x 5 columns]


Get the row mean of all columns::

    >>> DT[:, rowmean(f[:])]
       |      C0
       | float64
    -- + -------
     0 |     2.5
     1 |     3
     2 |     2
     3 |     0.2
    [4 rows x 1 column]

Get the row mean of specific columns::

    >>> DT[:, rowmean(f['a', 'b', 'd'])]
       |       C0
       |  float64
    -- + --------
     0 | 1
     1 | 2.33333
     2 | 2.66667
     3 | 0.333333
    [4 rows x 1 column]



See Also
--------
- :func:`rowsd()` -- calculate the standard deviation row-wise.
)";


const char* doc_dt_rowmin =
R"(rowmin(*cols)
--

For each row, find the smallest value among the columns from `cols`,
excluding missing values.


Parameters
----------
cols: FExpr
    Input columns.

return: FExpr
    f-expression consisting of one column that has the same number of rows
    as in `cols`. The column stype is the smallest common stype
    for `cols`, but not less than `int32`.

except: TypeError
    The exception is raised when `cols` has non-numeric columns.


Examples
--------
::

    >>> from datatable import dt, f
    >>> DT = dt.Frame({"A": [1, 1, 2, 1, 2],
    ...                "B": [None, 2, 3, 4, None],
    ...                "C":[True, False, False, True, True]})
    >>> DT
       |     A      B      C
       | int32  int32  bool8
    -- + -----  -----  -----
     0 |     1     NA      1
     1 |     1      2      0
     2 |     2      3      0
     3 |     1      4      1
     4 |     2     NA      1
    [5 rows x 3 columns]

::

    >>> DT[:, dt.rowmin(f[:])]
       |    C0
       | int32
    -- + -----
     0 |     1
     1 |     0
     2 |     0
     3 |     1
     4 |     1
    [5 rows x 1 column]


See Also
--------
- :func:`rowmax()` -- find the largest element row-wise.
)";


const char* doc_dt_rowsd =
R"(rowsd(*cols)
--

For each row, find the standard deviation among the columns from `cols`
skipping missing values. If a row contains only the missing values,
this function will produce a missing value too.


Parameters
----------
cols: FExpr
    Input columns.

return: FExpr
    f-expression consisting of one column that has the same number of rows
    as in `cols`. The column stype is `float32` when all the `cols`
    are `float32`, and `float64` in all the other cases.

except: TypeError
    The exception is raised when `cols` has non-numeric columns.


Examples
--------
::

    >>> from datatable import dt, f, rowsd
    >>> DT = dt.Frame({'name': ['A', 'B', 'C', 'D', 'E'],
    ...                'group': ['mn', 'mn', 'kl', 'kl', 'fh'],
    ...                'S1': [1, 4, 5, 6, 7],
    ...                'S2': [2, 3, 8, 5, 1],
    ...                'S3': [8, 5, 2, 5, 3]}

    >>> DT
       | name   group     S1     S2     S3
       | str32  str32  int32  int32  int32
    -- + -----  -----  -----  -----  -----
     0 | A      mn         1      2      8
     1 | B      mn         4      3      5
     2 | C      kl         5      8      2
     3 | D      kl         6      5      5
     4 | E      fh         7      1      3
    [5 rows x 5 columns]

Get the row standard deviation for all integer columns::

    >>> DT[:, rowsd(f[int])]
       |      C0
       | float64
    -- + -------
     0 | 3.78594
     1 | 1
     2 | 3
     3 | 0.57735
     4 | 3.05505
    [5 rows x 1 column]

Get the row standard deviation for some columns::

    >>> DT[:, rowsd(f[2, 3])]
       |       C0
       |  float64
    -- + --------
     0 | 0.707107
     1 | 0.707107
     2 | 2.12132
     3 | 0.707107
     4 | 4.24264
    [5 rows x 1 column]


See Also
--------
- :func:`rowmean()` -- calculate the mean value row-wise.
)";


const char* doc_dt_rowsum =
R"(rowsum(*cols)
--

For each row, calculate the sum of all values in `cols`. Missing values
are treated as if they are zeros and skipped during the calcultion.


Parameters
----------
cols: FExpr
    Input columns.

return: FExpr
    f-expression consisting of one column and the same number
    of rows as in `cols`. The stype of the resulting column
    will be the smallest common stype calculated for `cols`,
    but not less than `int32`.

except: TypeError
    The exception is raised when one of the columns from `cols`
    has a non-numeric type.

Examples
--------
::

    >>> from datatable import dt, f, rowsum
    >>> DT = dt.Frame({'a': [1,2,3],
    ...                'b': [2,3,4],
    ...                'c':['dd','ee','ff'],
    ...                'd':[5,9,1]})
    >>> DT
       |     a      b  c          d
       | int32  int32  str32  int32
    -- + -----  -----  -----  -----
     0 |     1      2  dd         5
     1 |     2      3  ee         9
     2 |     3      4  ff         1
    [3 rows x 4 columns]


::

    >>> DT[:, rowsum(f[int])]
       |    C0
       | int32
    -- + -----
     0 |     8
     1 |    14
     2 |     8
    [3 rows x 1 column]

::

    >>> DT[:, rowsum(f.a, f.b)]
       |    C0
       | int32
    -- + -----
     0 |     3
     1 |     5
     2 |     7
    [3 rows x 1 column]

The above code could also be written as ::

    >>> DT[:, f.a + f.b]
       |    C0
       | int32
    -- + -----
     0 |     3
     1 |     5
     2 |     7
    [3 rows x 1 column]



See Also
--------
- :func:`rowcount()` -- count non-missing values row-wise.
)";


const char* doc_dt_sd =
R"(sd(cols)
--

Calculate the standard deviation for each column from `cols`.


Parameters
----------
cols: FExpr
    Input columns.

return: Expr
    f-expression having one row, and the same names and number of columns
    as in `cols`. The column stypes are `float32` for
    `float32` columns, and `float64` for all the other numeric types.

except: TypeError
    The exception is raised when one of the columns from `cols`
    has a non-numeric type.


Examples
--------

.. code-block:: python

    >>> from datatable import dt, f
    >>>
    >>> DT = dt.Frame(A = [0, 1, 2, 3], B = [0, 2, 4, 6])
    >>> DT
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     0      0
     1 |     1      2
     2 |     2      4
     3 |     3      6
    [4 rows x 2 columns]


Get the standard deviation of column A::

    >>> DT[:, dt.sd(f.A)]
       |       A
       | float64
    -- + -------
     0 | 1.29099
    [1 row x 1 column]


Get the standard deviation of columns A and B::

    >>> DT[:, dt.sd([f.A, f.B])]
       |       A        B
       | float64  float64
    -- + -------  -------
     0 | 1.29099  2.58199
    [1 row x 2 columns]


See Also
--------

- :func:`mean()` -- function to calculate mean values.
- :func:`median()` -- function to calculate median values.
)";


const char* doc_dt_setdiff =
R"(setdiff(frame0, *frames)
--

Find the set difference between `frame0` and the other `frames`.

Each frame should have only a single column or be empty.
The values in each frame will be treated as a set, and this function will
compute the
`set difference  <https://en.wikipedia.org/wiki/Complement_(set_theory)#Relative_complement>`_
between the `frame0` and the union of the other
frames, returning those values that are present in the `frame0`,
but not present in any of the `frames`.

Parameters
----------
frame0: Frame
    Input single-column frame.

*frames: Frame, Frame, ...
    Input single-column frames.

return: Frame
    A single-column frame. The column stype is the smallest common
    stype of columns from the `frames`.

except: ValueError | NotImplementedError
    .. list-table::
        :widths: auto
        :class: api-table

        * - :exc:`dt.exceptions.ValueError`
          - raised when one of the input frames, i.e. `frame0`
            or any one from the `frames`, has more than one column.

        * - :exc:`dt.exceptions.NotImplementedError`
          - raised when one of the columns has stype `obj64`.

Examples
--------
.. code-block:: python

    >>> from datatable import dt
    >>>
    >>> s1 = dt.Frame([4, 5, 6, 20, 42])
    >>> s2 = dt.Frame([1, 2, 3, 5, 42])
    >>>
    >>> s1
       |    C0
       | int32
    -- + -----
     0 |     4
     1 |     5
     2 |     6
     3 |    20
     4 |    42
    [5 rows x 1 column]

    >>> s2
       |    C0
       | int32
    -- + -----
     0 |     1
     1 |     2
     2 |     3
     3 |     5
     4 |    42
    [5 rows x 1 column]

Set difference of the two frames::

    >>> dt.setdiff(s1, s2)
       |    C0
       | int32
    -- + -----
     0 |     4
     1 |     6
     2 |    20
    [3 rows x 1 column]



See Also
--------
- :func:`intersect()` -- calculate the set intersection of values in the frames.
- :func:`symdiff()` -- calculate the symmetric difference between the sets of values in the frames.
- :func:`union()` -- calculate the union of values in the frames.
- :func:`unique()` -- find unique values in a frame.
)";


const char* doc_dt_shift =
R"(shift(cols, n=1)
--

Shift columns' data forward or backwards. This function is group-aware,
i.e. in the presence of a groupby it will shift data separately
within each group.


Parameters
----------
cols: FExpr
    Input data to be shifted.

n: int
    The shift amount. For positive `n`, the data are shifted forward, i.e.
    a "lag" column is created. For negative `n`, the data are shifted backwards
    creating a "lead" column.

return: FExpr
    f-expression that shifts input data by `n` rows. The resulting data
    will have the same number of rows as `cols`. Depending on `n`,
    i.e. positive/negative, `n` observations in the beginning/end
    will become missing and `n` observations at the end/beginning
    will be discarded.


Examples
--------
.. code-block:: python

    >>> from datatable import dt, f, by
    >>>
    >>> DT = dt.Frame({"object": [1, 1, 1, 2, 2],
    >>>                "period": [1, 2, 4, 4, 23],
    >>>                "value": [24, 67, 89, 5, 23]})
    >>>
    >>> DT
       | object  period  value
       |  int32   int32  int32
    -- + ------  ------  -----
     0 |      1       1     24
     1 |      1       2     67
     2 |      1       4     89
     3 |      2       4      5
     4 |      2      23     23
    [5 rows x 3 columns]

Shift forward by creating a "lag" column::

    >>> DT[:, dt.shift(f.period,  n = 3)]
       | period
       |  int32
    -- + ------
     0 |     NA
     1 |     NA
     2 |     NA
     3 |      1
     4 |      2
    [5 rows x 1 column]

Shift backwards by creating "lead" columns::

    >>> DT[:, dt.shift(f[:], n = -3)]
       | object  period  value
       |  int32   int32  int32
    -- + ------  ------  -----
     0 |      2       4      5
     1 |      2      23     23
     2 |     NA      NA     NA
     3 |     NA      NA     NA
     4 |     NA      NA     NA
    [5 rows x 3 columns]

Shift within groups, i.e. in the presence of :func:`by()`::

    >>> DT[:, f[:].extend({"prev_value": dt.shift(f.value)}), by("object")]
       | object  period  value  prev_value
       |  int32   int32  int32       int32
    -- + ------  ------  -----  ----------
     0 |      1       1     24          NA
     1 |      1       2     67          24
     2 |      1       4     89          67
     3 |      2       4      5          NA
     4 |      2      23     23           5
    [5 rows x 4 columns]
)";


const char* doc_dt_sort =
R"(sort(*cols, reverse=False, na_position="first")
--

Sort clause for use in Frame's square-bracket selector.

When a ``sort()`` object is present inside a ``DT[i, j, ...]``
expression, it will sort the rows of the resulting Frame according
to the columns ``cols`` passed as the arguments to ``sort()``.

When used together with ``by()``, the sort clause applies after the
group-by, i.e. we sort elements within each group. Note, however,
that because we use stable sorting, the operations of grouping and
sorting are commutative: the result of applying groupby and then sort
is the same as the result of sorting first and then doing groupby.

When used together with ``i`` (row filter), the ``i`` filter is
applied after the sorting. For example::

    >>> DT[:10, :, sort(f.Highscore, reverse=True)]

will select the first 10 records from the frame ``DT`` ordered by
the Highscore column.


Parameters
----------
cols: FExpr
    Columns to sort the frame by.

reverse: bool
    If ``False``, sorting is performed in the ascending order. If ``True``,
    sorting is descending.

na_position: None | "first" | "last" | "remove"
    If "first", the default behaviour, missing values are placed
    first in sorted output. If "last", they are placed last.
    If "remove", rows that contain missing values in `cols`
    are excluded from the output.

return: object
    ``datatable.sort`` object for use in square-bracket selector.


Examples
--------
.. code-block:: python

    >>> from datatable import dt, f, by
    >>>
    >>> DT = dt.Frame({"col1": ["A", "A", "B", None, "D", "C"],
    >>>                "col2": [2, 1, 9, 8, 7, 4],
    >>>                "col3": [0, 1, 9, 4, 2, 3],
    >>>                "col4": [1, 2, 3, 3, 2, 1]})
    >>>
    >>> DT
       | col1    col2   col3   col4
       | str32  int32  int32  int32
    -- + -----  -----  -----  -----
     0 | A          2      0      1
     1 | A          1      1      2
     2 | B          9      9      3
     3 | NA         8      4      3
     4 | D          7      2      2
     5 | C          4      3      1
    [6 rows x 4 columns]

Sort by a single column::

    >>> DT[:, :, dt.sort("col1")]
       | col1    col2   col3   col4
       | str32  int32  int32  int32
    -- + -----  -----  -----  -----
     0 | NA         8      4      3
     1 | A          2      0      1
     2 | A          1      1      2
     3 | B          9      9      3
     4 | C          4      3      1
     5 | D          7      2      2
    [6 rows x 4 columns]


Sort by multiple columns::

    >>> DT[:, :, dt.sort("col2", "col3")]
       | col1    col2   col3   col4
       | str32  int32  int32  int32
    -- + -----  -----  -----  -----
     0 | A          1      1      2
     1 | A          2      0      1
     2 | C          4      3      1
     3 | D          7      2      2
     4 | NA         8      4      3
     5 | B          9      9      3
    [6 rows x 4 columns]

Sort in descending order::

    >>> DT[:, :, dt.sort(-f.col1)]
       | col1    col2   col3   col4
       | str32  int32  int32  int32
    -- + -----  -----  -----  -----
     0 | NA         8      4      3
     1 | D          7      2      2
     2 | C          4      3      1
     3 | B          9      9      3
     4 | A          2      0      1
     5 | A          1      1      2
    [6 rows x 4 columns]

The frame can also be sorted in descending order by setting the ``reverse`` parameter to ``True``::

    >>> DT[:, :, dt.sort("col1", reverse=True)]
       | col1    col2   col3   col4
       | str32  int32  int32  int32
    -- + -----  -----  -----  -----
     0 | NA         8      4      3
     1 | D          7      2      2
     2 | C          4      3      1
     3 | B          9      9      3
     4 | A          2      0      1
     5 | A          1      1      2
    [6 rows x 4 columns]

By default, when sorting, null values are placed at the top; to relocate null values to the bottom, pass ``last`` to the ``na_position`` parameter::

    >>> DT[:, :, dt.sort("col1", na_position="last")]
       | col1    col2   col3   col4
       | str32  int32  int32  int32
    -- + -----  -----  -----  -----
     0 | A          2      0      1
     1 | A          1      1      2
     2 | B          9      9      3
     3 | C          4      3      1
     4 | D          7      2      2
     5 | NA         8      4      3
    [6 rows x 4 columns]

Passing ``remove`` to ``na_position`` completely excludes any row with null values from the sorted output:

    >>> DT[:, :, dt.sort("col1", na_position="remove")]
       | col1    col2   col3   col4
       | str32  int32  int32  int32
    -- + -----  -----  -----  -----
     0 | A          2      0      1
     1 | A          1      1      2
     2 | B          9      9      3
     3 | C          4      3      1
     4 | D          7      2      2
    [5 rows x 4 columns]

Sort by multiple columns, descending and ascending order::

    >>> DT[:, :, dt.sort(-f.col2, f.col3)]
       | col1    col2   col3   col4
       | str32  int32  int32  int32
    -- + -----  -----  -----  -----
     0 | B          9      9      3
     1 | NA         8      4      3
     2 | D          7      2      2
     3 | C          4      3      1
     4 | A          2      0      1
     5 | A          1      1      2
    [6 rows x 4 columns]

The same code above can be replicated by passing a list of booleans to ``reverse``. 
The length of the reverse flag list should match the number of columns to be sorted::

    >>> DT[:, :, dt.sort("col2", "col3", reverse=[True, False])]
       | col1    col2   col3   col4
       | str32  int32  int32  int32
    -- + -----  -----  -----  -----
     0 | B          9      9      3
     1 | NA         8      4      3
     2 | D          7      2      2
     3 | C          4      3      1
     4 | A          2      0      1
     5 | A          1      1      2
    [6 rows x 4 columns]

In the presence of :func:`by()`, :func:`sort()` sorts within each group::

    >>> DT[:, :, by("col4"), dt.sort(f.col2)]
       |  col4  col1    col2   col3
       | int32  str32  int32  int32
    -- + -----  -----  -----  -----
     0 |     1  A          2      0
     1 |     1  C          4      3
     2 |     2  A          1      1
     3 |     2  D          7      2
     4 |     3  NA         8      4
     5 |     3  B          9      9
    [6 rows x 4 columns]
)";


const char* doc_dt_split_into_nhot =
R"(split_into_nhot(frame, sep=",", sort=False)
--

.. x-version-deprecated:: 1.0.0

    This function is deprecated and will be removed in version 1.1.0.
    Please use :func:`dt.str.split_into_nhot()` instead.
)";


const char* doc_dt_sum =
R"(sum(cols)
--

Calculate the sum of values for each column from `cols`. The sum of
the missing values is calculated as zero.


Parameters
----------
cols: FExpr
    Input columns.

return: Expr
    f-expression having one row, and the same names and number of columns
    as in `cols`. The column types are :attr:`int64 <dt.Type.int64>` for
    void, boolean and integer columns, :attr:`float32 <dt.Type.float32>`
    for :attr:`float32 <dt.Type.float32>` columns and
    :attr:`float64 <dt.Type.float64>` for :attr:`float64 <dt.Type.float64>`
    columns.

except: TypeError
    The exception is raised when one of the columns from `cols`
    has a non-numeric type.


Examples
--------

.. code-block:: python

    >>> from datatable import dt, f, by
    >>>
    >>> df = dt.Frame({'A': [1, 1, 2, 1, 2],
    ...                'B': [None, 2, 3, 4, 5],
    ...                'C': [1, 2, 1, 1, 2]})
    >>> df
       |     A      B      C
       | int32  int32  int32
    -- + -----  -----  -----
     0 |     1     NA      1
     1 |     1      2      2
     2 |     2      3      1
     3 |     1      4      1
     4 |     2      5      2
    [5 rows x 3 columns]

Get the sum of column A::

    >>> df[:, dt.sum(f.A)]
       |     A
       | int64
    -- + -----
     0 |     7
    [1 row x 1 column]

Get the sum of multiple columns::

    >>> df[:, [dt.sum(f.A), dt.sum(f.B)]]
       |     A      B
       | int64  int64
    -- + -----  -----
     0 |     7     14
    [1 row x 2 columns]

Same as above, but more convenient::

    >>> df[:, dt.sum(f[:2])]
       |     A      B
       | int64  int64
    -- + -----  -----
     0 |     7     14
    [1 row x 2 columns]

In the presence of :func:`by()`, it returns the sum of the specified columns per group::

    >>> df[:, [dt.sum(f.A), dt.sum(f.B)], by(f.C)]
       |     C      A      B
       | int32  int64  int64
    -- + -----  -----  -----
     0 |     1      4      7
     1 |     2      3      7
    [2 rows x 3 columns]


See Also
--------
- :meth:`.sum() <dt.Frame.sum>` -- the corresponding Frame's method
  for multi-column frames.

- :meth:`.sum1() <dt.Frame.sum1>` -- the corresponding Frame's method
  for single-column frames that returns a scalar value.

- :func:`count()` -- function to calculate a number of non-missing values.
)";


const char* doc_dt_symdiff =
R"(symdiff(*frames)
--

Find the symmetric difference between the sets of values in all `frames`.

Each frame should have only a single column or be empty.
The values in each frame will be treated as a set, and this function will
perform the
`symmetric difference <https://en.wikipedia.org/wiki/Symmetric_difference>`_
operation on these sets.

The symmetric difference of two frames are those values that are present in
either of the frames, but not in the both. The symmetric difference of more than
two frames are those values that are present in an odd number of frames.

Parameters
----------
*frames: Frame, Frame, ...
    Input single-column frames.

return: Frame
    A single-column frame. The column stype is the smallest common
    stype of columns from the `frames`.

except: ValueError | NotImplementedError
    .. list-table::
        :widths: auto
        :class: api-table

        * - :exc:`dt.exceptions.ValueError`
          - raised when one of the input frames has more than one column.

        * - :exc:`dt.exceptions.NotImplementedError`
          - raised when one of the columns has stype `obj64`.

Examples
--------
.. code-block:: python

    >>> from datatable import dt
    >>>
    >>> df = dt.Frame({'A': [1, 1, 2, 1, 2],
    ...                'B': [None, 2, 3, 4, 5],
    ...                'C': [1, 2, 1, 1, 2]})
    >>> df
       |     A      B      C
       | int32  int32  int32
    -- + -----  -----  -----
     0 |     1     NA      1
     1 |     1      2      2
     2 |     2      3      1
     3 |     1      4      1
     4 |     2      5      2
    [5 rows x 3 columns]

Symmetric difference of all the columns in the entire frame; Note that each column is treated as a separate frame::

    >>> dt.symdiff(*df)
       |     A
       | int32
    -- + -----
     0 |    NA
     1 |     2
     2 |     3
     3 |     4
     4 |     5
    [5 rows x 1 column]


Symmetric difference between two frames::

    >>> dt.symdiff(df["A"], df["B"])
       |     A
       | int32
    -- + -----
     0 |    NA
     1 |     1
     2 |     3
     3 |     4
     4 |     5
    [5 rows x 1 column]


See Also
--------
- :func:`intersect()` -- calculate the set intersection of values in the frames.
- :func:`setdiff()` -- calculate the set difference between the frames.
- :func:`union()` -- calculate the union of values in the frames.
- :func:`unique()` -- find unique values in a frame.
)";


const char* doc_dt_union =
R"(union(*frames)
--

Find the union of values in all `frames`.

Each frame should have only a single column or be empty. The values in each
frame will be treated as a set, and this function will perform the
`union operation <https://en.wikipedia.org/wiki/Union_(set_theory)>`_
on these sets.

The `dt.union(*frames)` operation is equivalent to
`dt.unique(dt.rbind(*frames))`.

Parameters
----------
*frames: Frame, Frame, ...
    Input single-column frames.

return: Frame
    A single-column frame. The column stype is the smallest common
    stype of columns in the `frames`.

except: ValueError | NotImplementedError
    .. list-table::
        :widths: auto
        :class: api-table

        * - :exc:`dt.exceptions.ValueError`
          - raised when one of the input frames has more than one column.

        * - :exc:`dt.exceptions.NotImplementedError`
          - raised when one of the columns has stype `obj64`.


Examples
--------
.. code-block:: python

    >>> from datatable import dt
    >>>
    >>> df = dt.Frame({'A': [1, 1, 2, 1, 2],
    ...                'B': [None, 2, 3, 4, 5],
    ...                'C': [1, 2, 1, 1, 2]})
    >>> df
       |     A      B      C
       | int32  int32  int32
    -- + -----  -----  -----
     0 |     1     NA      1
     1 |     1      2      2
     2 |     2      3      1
     3 |     1      4      1
     4 |     2      5      2
    [5 rows x 3 columns]

Union of all the columns in a frame::

    >>> dt.union(*df)
       |     A
       | int32
    -- + -----
     0 |    NA
     1 |     1
     2 |     2
     3 |     3
     4 |     4
     5 |     5
    [6 rows x 1 column]

Union of two frames::

    >>> dt.union(df["A"], df["C"])
       |     A
       | int32
    -- + -----
     0 |     1
     1 |     2
    [2 rows x 1 column]


See Also
--------
- :func:`intersect()` -- calculate the set intersection of values in the frames.
- :func:`setdiff()` -- calculate the set difference between the frames.
- :func:`symdiff()` -- calculate the symmetric difference between the sets of values in the frames.
- :func:`unique()` -- find unique values in a frame.
)";


const char* doc_dt_unique =
R"(unique(frame)
--

Find the unique values in all the columns of the `frame`.

This function sorts the values in order to find the uniques, so the return
values will be ordered. However, this should be considered an implementation
detail: in the future datatable may switch to a different algorithm,
such as hash-based, which may return the results in a different order.


Parameters
----------
frame: Frame
    Input frame.

return: Frame
    A single-column frame consisting of unique values found in `frame`.
    The column stype is the smallest common stype for all the
    `frame` columns.

except: NotImplementedError
    The exception is raised when one of the frame columns has stype `obj64`.


Examples
--------
.. code-block:: python

    >>> from datatable import dt
    >>>
    >>> df = dt.Frame({'A': [1, 1, 2, 1, 2],
    ...                'B': [None, 2, 3, 4, 5],
    ...                'C': [1, 2, 1, 1, 2]})
    >>> df
       |     A      B      C
       | int32  int32  int32
    -- + -----  -----  -----
     0 |     1     NA      1
     1 |     1      2      2
     2 |     2      3      1
     3 |     1      4      1
     4 |     2      5      2
    [5 rows x 3 columns]

Unique values in the entire frame::

    >>> dt.unique(df)
       |    C0
       | int32
    -- + -----
     0 |    NA
     1 |     1
     2 |     2
     3 |     3
     4 |     4
     5 |     5
    [6 rows x 1 column]

Unique values in a frame with a single column::

    >>> dt.unique(df["A"])
       |     A
       | int32
    -- + -----
     0 |     1
     1 |     2
    [2 rows x 1 column]


See Also
--------
- :func:`intersect()` -- calculate the set intersection of values in the frames.
- :func:`setdiff()` -- calculate the set difference between the frames.
- :func:`symdiff()` -- calculate the symmetric difference between the sets of values in the frames.
- :func:`union()` -- calculate the union of values in the frames.
)";


const char* doc_dt_update =
R"(update(**kwargs)
--

Create new or update existing columns within a frame.

This expression is intended to be used at "j" place in ``DT[i, j]``
call. It takes an arbitrary number of key/value pairs each describing
a column name and the expression for how that column has to be
created/updated.

Examples
--------
.. code-block:: python

    >>> from datatable import dt, f, by, update
    >>>
    >>> DT = dt.Frame([range(5), [4, 3, 9, 11, -1]], names=("A", "B"))
    >>> DT
       |     A      B
       | int32  int32
    -- + -----  -----
     0 |     0      4
     1 |     1      3
     2 |     2      9
     3 |     3     11
     4 |     4     -1
    [5 rows x 2 columns]

Create new columns and update existing columns::

    >>> DT[:, update(C = f.A * 2,
    ...              D = f.B // 3,
    ...              A = f.A * 4,
    ...              B = f.B + 1)]
    >>> DT
       |     A      B      C      D
       | int32  int32  int32  int32
    -- + -----  -----  -----  -----
     0 |     0      5      0      1
     1 |     4      4      2      1
     2 |     8     10      4      3
     3 |    12     12      6      3
     4 |    16      0      8     -1
    [5 rows x 4 columns]

Add new column with `unpacking`_; this can be handy for dynamicallly adding
columns with dictionary comprehensions, or if the names are not valid python
keywords::

    >>> DT[:, update(**{"extra column": f.A + f.B + f.C + f.D})]
    >>> DT
       |     A      B      C      D  extra column
       | int32  int32  int32  int32         int32
    -- + -----  -----  -----  -----  ------------
     0 |     0      5      0      1             6
     1 |     4      4      2      1            11
     2 |     8     10      4      3            25
     3 |    12     12      6      3            33
     4 |    16      0      8     -1            23
    [5 rows x 5 columns]

You can update a subset of data::

    >>> DT[f.A > 10, update(A = f.A * 5)]
    >>> DT
       |     A      B      C      D  extra column
       | int32  int32  int32  int32         int32
    -- + -----  -----  -----  -----  ------------
     0 |     0      5      0      1             6
     1 |     4      4      2      1            11
     2 |     8     10      4      3            25
     3 |    60     12      6      3            33
     4 |    80      0      8     -1            23
    [5 rows x 5 columns]

You can also add a new column or update an existing column in a groupby
operation, similar to SQL's `window` operation, or pandas `transform()`::

    >>> df = dt.Frame("""exporter assets   liabilities
    ...                   False      5          1
    ...                   True       10         8
    ...                   False      3          1
    ...                   False      24         20
    ...                   False      40         2
    ...                   True       12         11""")
    >>>
    >>> # Get the ratio for each row per group
    >>> df[:,
    ...    update(ratio = dt.sum(f.liabilities) * 100 / dt.sum(f.assets)),
    ...    by(f.exporter)]
    >>> df
       | exporter  assets  liabilities    ratio
       |    bool8   int32        int32  float64
    -- + --------  ------  -----------  -------
     0 |        0       5            1  33.3333
     1 |        1      10            8  86.3636
     2 |        0       3            1  33.3333
     3 |        0      24           20  33.3333
     4 |        0      40            2  33.3333
     5 |        1      12           11  86.3636
    [6 rows x 4 columns]


.. _`unpacking` : https://docs.python.org/3/tutorial/controlflow.html#unpacking-argument-lists
)";


const char* doc_dt_prod =
R"(prod(cols)
--

.. x-version-added:: 1.1.0

Calculate the product of values for each column from `cols`. The product
of missing values is calculated as one.

Parameters
----------
cols: FExpr
    Input columns.

return: Expr
    f-expression having one row, and the same names and number of columns
    as in `cols`. The column stypes are `int64` for
    boolean and integer columns, `float32` for `float32` columns
    and `float64` for `float64` columns.

except: TypeError
    The exception is raised when one of the columns from `cols`
    has a non-numeric type.


Examples
--------

.. code-block:: python

    >>> from datatable import dt, f, by, prod
    >>>
    >>> DT = dt.Frame({'A': [1, 1, 2, 1, 2],
    ...                'B': [None, 2, 3, 4, 5],
    ...                'C': [1, 2, 1, 1, 2]})
    >>> DT
       |     A      B      C
       | int32  int32  int32
    -- + -----  -----  -----
     0 |     1     NA      1
     1 |     1      2      2
     2 |     2      3      1
     3 |     1      4      1
     4 |     2      5      2
    [5 rows x 3 columns]

Get the product for column A::

    >>> DT[:, prod(f.A)]
       |     A
       | int64
    -- + -----
     0 |     4
    [1 row x 1 column]

Get the product for multiple columns::

    >>> DT[:, prod(f['A', 'B'])]
       |     A      B
       | int64  int64
    -- + -----  -----
     0 |     4    120
    [1 row x 2 columns]


In the presence of :func:`by()`, it returns the product of the specified columns per group::

    >>> DT[:, prod(f['A', 'B']), by('C')]
       |     C      A      B
       | int32  int64  int64
    -- + -----  -----  -----
     0 |     1      2     12
     1 |     2      2     10
    [2 rows x 3 columns]
)";


const char* doc_internal_frame_column_data_r =
R"(frame_column_data_r(frame, i)
--

Return C pointer to the main data array of the column `frame[i]`.
The column will be materialized if it was virtual.


Parameters
----------
frame: Frame
    The :class:`dt.Frame` where to look up the column.

i: int
    The index of a column, in the range ``[0; ncols)``.

return: ctypes.c_void_p
    The pointer to the column's internal data.
)";


const char* doc_internal_frame_columns_virtual =
R"(frame_columns_virtual(frame)
--

.. x-version-deprecated:: 0.11.0

Return the list indicating which columns in the `frame` are virtual.

Parameters
----------
return: List[bool]
    Each element in the list indicates whether the corresponding column
    is virtual or not.

Notes
-----

This function will be expanded and moved into the main :class:`dt.Frame` class.
)";


const char* doc_internal_frame_integrity_check =
R"(frame_integrity_check(frame)
--

This function performs a range of tests on the `frame` to verify
that its internal state is consistent. It returns None on success,
or throws an ``AssertionError`` if any problems were found.


Parameters
----------
frame: Frame
    A :class:`dt.Frame` object that needs to be checked for internal consistency.

return: None

except: AssertionError
    An exception is raised if there were any issues with the `frame`.
)";


const char* doc_internal_get_thread_ids =
R"(get_thread_ids()
--

Return system ids of all threads used internally by datatable.

Calling this function will cause the threads to spawn if they
haven't done already. (This behavior may change in the future).


Parameters
----------
return: List[str]
    The list of thread ids used by the datatable. The first element
    in the list is the id of the main thread.


See Also
--------
- :attr:`dt.options.nthreads` -- global option that controls the
  number of threads in use.
)";


const char* doc_math_abs =
R"(abs(x)
--

Return the absolute value of ``x``. This function can only be applied
to numeric arguments (i.e. boolean, integer, or real).

This function upcasts columns of types `bool8`, `int8` and `int16` into
`int32`; for columns of other types the stype is kept.

Parameters
----------
x: FExpr
    Column expression producing one or more numeric columns.

return: FExpr
    The resulting FExpr evaluates absolute values in all elements
    in all columns of `x`.

Examples
--------

.. code-block::

    >>> DT = dt.Frame(A=[-3, 2, 4, -17, 0])
    >>> DT[:, abs(f.A)]
       |     A
       | int32
    -- + -----
     0 |     3
     1 |     2
     2 |     4
     3 |    17
     4 |     0
    [5 rows x 1 column]

See also
--------
- :func:`fabs()`
)";


const char* doc_math_arccos =
R"(arccos(x)
--

Inverse trigonometric cosine of `x`.

In mathematics, this may be written as :math:`\arccos x` or
:math:`\cos^{-1}x`.

The returned value is in the interval :math:`[0, \frac12\tau]`,
and NA for the values of ``x`` that lie outside the interval
``[-1, 1]``. This function is the inverse of
:func:`cos()` in the sense that
`cos(arccos(x)) == x` for all ``x`` in the interval ``[-1, 1]``.

See also
--------
- :func:`cos(x)` -- the trigonometric cosine function;
- :func:`arcsin(x)` -- the inverse sine function.
)";


const char* doc_math_arcosh =
R"(arcosh(x)
--

The inverse hyperbolic cosine of `x`.

This function satisfies the property that ``cosh(arccosh(x)) == x``.
Alternatively, this function can also be computed as
:math:`\cosh^{-1}(x) = \ln(x + \sqrt{x^2 - 1})`.

See also
--------
- :func:`cosh` -- hyperbolic cosine;
- :func:`arsinh` -- inverse hyperbolic sine.
)";


const char* doc_math_arcsin =
R"(arcsin(x)
--

Inverse trigonometric sine of `x`.

In mathematics, this may be written as :math:`\arcsin x` or
:math:`\sin^{-1}x`.

The returned value is in the interval :math:`[-\frac14 \tau, \frac14\tau]`,
and NA for the values of ``x`` that lie outside the interval ``[-1, 1]``.
This function is the inverse of :func:`sin()` in the sense
that `sin(arcsin(x)) == x` for all ``x`` in the interval ``[-1, 1]``.

See also
--------
- :func:`sin(x)` -- the trigonometric sine function;
- :func:`arccos(x)` -- the inverse cosine function.
)";


const char* doc_math_arctan =
R"(arctan(x)
--

Inverse trigonometric tangent of `x`.

This function satisfies the property that ``tan(arctan(x)) == x``.

See also
--------
- :func:`atan2(x, y)` -- two-argument inverse tangent function;
- :func:`tan(x)` -- the trigonometric tangent function.
)";


const char* doc_math_arsinh =
R"(arsinh(x)
--

The inverse hyperbolic sine of `x`.

This function satisfies the property that ``sinh(arcsinh(x)) == x``.
Alternatively, this function can also be computed as
:math:`\sinh^{-1}(x) = \ln(x + \sqrt{x^2 + 1})`.

See also
--------
- :func:`sinh` -- hyperbolic sine;
- :func:`arcosh` -- inverse hyperbolic cosine.
)";


const char* doc_math_artanh =
R"(artanh(x)
--

The inverse hyperbolic tangent of `x`.

This function satisfies the property that ``tanh(artanh(x)) == x``.
Alternatively, this function can also be computed as
:math:`\tanh^{-1}(x) = \frac12\ln\frac{1+x}{1-x}`.

See also
--------
- :func:`tanh` -- hyperbolic tangent;
)";


const char* doc_math_atan2 =
R"(atan2(x, y)
--

The inverse trigonometric tangent of ``y/x``, taking into account the signs
of ``x`` and ``y`` to produce the correct result.


If ``(x,y)`` is a point in a Cartesian plane, then ``arctan2(y, x)`` returns
the radian measure of an angle formed by two rays: one starting at the origin
and passing through point ``(0,1)``, and the other starting at the origin
and passing through point ``(x,y)``. The angle is assumed positive if the
rotation from the first ray to the second occurs counter-clockwise, and
negative otherwise.

As a special case, ``arctan2(0, 0) == 0``, and ``arctan2(0, -1) == tau/2``.


See also
--------
- :func:`arctan(x)` -- inverse tangent function.
)";


const char* doc_math_cbrt =
R"(cbrt(x)
--

Cubic root of x.
)";


const char* doc_math_ceil =
R"(ceil(x)
--

The smallest integer value not less than `x`, returned as float.

This function produces a ``float32`` column if the input is of type
``float32``, or ``float64`` columns for inputs of all other numeric
stypes.

Parameters
----------
x: FExpr
    One or more numeric columns.

return: FExpr
    Expression that computes the ``ceil()`` function for each row and
    column in `x`.
)";


const char* doc_math_copysign =
R"(copysign(x, y)
--

Return a float with the magnitude of ``x`` and the sign of ``y``.
)";


const char* doc_math_cos =
R"(cos(x)
--

Compute the trigonometric cosine of angle ``x`` measured in radians.

This function can only be applied to numeric columns (real, integer, or
boolean), and produces a float64 result, except when the argument ``x`` is
float32, in which case the result is float32 as well.

See also
--------
- :func:`sin(x)` -- the trigonometric sine function;
- :func:`arccos(x)` -- the inverse cosine function.
)";


const char* doc_math_cosh =
R"(cosh(x)
--

The hyperbolic cosine of `x`, defined as
:math:`\cosh x = \frac12(e^x + e^{-x})`.

See also
--------
- :func:`sinh` -- hyperbolic sine;
- :func:`arcosh` -- inverse hyperbolic cosine.
)";


const char* doc_math_deg2rad =
R"(deg2rad(x)
--

Convert angle measured in degrees into radians:
:math:`\operatorname{deg2rad}(x) = x\cdot\frac{\tau}{360}`.

See also
--------
- :func:`rad2deg(x)` -- convert radians into degrees.
)";


const char* doc_math_erf =
R"(erf(x)
--

Error function ``erf(x)``, which is defined as the integral

.. math::

    \operatorname{erf}(x) = \frac{2}{\sqrt{\tau}} \int^{x/\sqrt{2}}_0 e^{-\frac12 t^2}dt

This function is used in computing probabilities arising from the normal
distribution.

See also
--------
- :func:`erfc(x) <datatable.math.erfc>` -- complimentary error function.
)";


const char* doc_math_erfc =
R"(erfc(x)
--

Complementary error function ``erfc(x) = 1 - erf(x)``.

The complementary error function is defined as the integral

.. math::

    \operatorname{erfc}(x) = \frac{2}{\sqrt{\tau}} \int^{\infty}_{x/\sqrt{2}} e^{-\frac12 t^2}dt

Although mathematically `erfc(x) = 1-erf(x)`, in practice the RHS
suffers catastrophic loss of precision at large values of `x`. This
function, however, does not have such a drawback.

See also
--------
- :func:`erf(x) <datatable.math.erf>` -- the error function.
)";


const char* doc_math_exp2 =
R"(exp2(x)
--

Binary exponent of x, same as :math:`2^x`.

See also
--------
- :func:`exp(x)` -- base-:math:`e` exponent.
)";


const char* doc_math_exp =
R"(exp(x)
--

The exponent of `x`, that is :math:`e^x`.

See also
--------
- :data:`e` -- the Euler's number;
- :func:`expm1(x)` -- exponent function minus one;
- :func:`exp2(x)` -- binary exponent;
)";


const char* doc_math_expm1 =
R"(expm1(x)
--

The exponent of `x` minus 1, that is :math:`e^x - 1`. This function is
more accurate for arguments `x` close to zero.
)";


const char* doc_math_fabs =
R"(fabs(x)
--

The absolute value of `x`, returned as float.
)";


const char* doc_math_floor =
R"(floor(x)
--

The largest integer value not greater than `x`, returned as float.

This function produces a ``float32`` column if the input is of type
``float32``, or ``float64`` columns for inputs of all other numeric
stypes.

Parameters
----------
x: FExpr
    One or more numeric columns.

return: FExpr
    Expression that computes the ``floor()`` function for each row and
    column in `x`.
)";


const char* doc_math_fmod =
R"(fmod(x, y)
--

Floating-point remainder of the division x/y. The result is always
a float, even if the arguments are integers. This function uses
``std::fmod()`` from the standard C++ library, its convention for
handling of negative numbers may be different than the Python's.
)";


const char* doc_math_gamma =
R"(gamma(x)
--

Euler Gamma function of x.

The gamma function is defined for all ``x`` except for the negative
integers. For positive ``x`` it can be computed via the integral

.. math::
    \Gamma(x) = \int_0^\infty t^{x-1}e^{-t}dt

For negative ``x`` it can be computed as

.. math::
    \Gamma(x) = \frac{\Gamma(x + k)}{x(x+1)\cdot...\cdot(x+k-1)}

where :math:`k` is any integer such that :math:`x+k` is positive.

If `x` is a positive integer, then :math:`\Gamma(x) = (x - 1)!`.

See also
--------
- :func:`lgamma(x) <datatable.math.lgamma>` -- log-gamma function.
)";


const char* doc_math_hypot =
R"(hypot(x, y)
--

The length of the hypotenuse of a right triangle with sides `x` and `y`,
or in math notation :math:`\operatorname{hypot}(x, y) = \sqrt{x^2 + y^2}`.
)";


const char* doc_math_isclose =
R"(isclose(x, y, *, rtol=1e-5, atol=1e-8)
--

Compare two numbers x and y, and return True if they are close
within the requested relative/absolute tolerance. This function
only returns True/False, never NA.

More specifically, isclose(x, y) is True if either of the following
are true:

- ``x == y`` (including the case when x and y are NAs),
- ``abs(x - y) <= atol + rtol * abs(y)`` and neither x nor y are NA

The tolerance parameters ``rtol``, ``atol`` must be positive floats,
and cannot be expressions.
)";


const char* doc_math_isfinite =
R"(isfinite(x)
--

Returns True if `x` has a finite value, and False if `x` is infinity
or NaN. This function is equivalent to ``!(isna(x) or isinf(x))``.

See also
--------
- :func:`isna()`
- :func:`isinf()`
)";


const char* doc_math_isinf =
R"(isinf(x)
--

Returns True if the argument is +/- infinity, and False otherwise.
Note that `isinf(NA) == False`.
)";


const char* doc_math_isna =
R"(isna(x)
--

Returns `True` if the argument is NA, and `False` otherwise.
)";


const char* doc_math_ldexp =
R"(ldexp(x, y)
--

Multiply ``x`` by 2 raised to the power ``y``, i.e. compute ``x * 2**y``.
Column ``x`` is expected to be float, and ``y`` integer.
)";


const char* doc_math_lgamma =
R"(lgamma(x)
--

Natural logarithm of the absolute value of the Euler Gamma
function of `x`.
)";


const char* doc_math_log10 =
R"(log10(x)
--

Decimal (base-10) logarithm of x, which is :math:`\lg(x)` or
:math:`\log_{10} x`. This function is the inverse of
:func:`pow(10, x)`.

See also
--------
- :func:`log()` -- natural logarithm;
- :func:`log2()` -- binary logarithm.
)";


const char* doc_math_log1p =
R"(log1p(x)
--

Natural logarithm of 1 plus `x`, or :math:`\ln(1 + x)`. This function
has improved numeric precision for small values of `x`.
)";


const char* doc_math_log2 =
R"(log2(x)
--

Binary (base-2) logarithm of x, which in mathematics is :math:`\log_2 x`.

See also
--------
- :func:`log()` -- natural logarithm;
- :func:`log10()` -- decimal logarithm.
)";


const char* doc_math_log =
R"(log(x)
--

Natural logarithm of `x`, aka :math:`\ln x`. This function is the
inverse of :func:`exp()`.

See also
--------
- :func:`log10()` -- decimal logarithm;
- :func:`log2()` -- binary logarithm.
)";


const char* doc_math_logaddexp2 =
R"(logaddexp2(x, y)
--

Binary logarithm of the sum of binary exponents of ``x`` and ``y``. This
function is equivalent to ``log2(exp2(x) + exp2(y))``, but does
not suffer from catastrophic precision loss for small values of
``x`` and ``y``.
)";


const char* doc_math_logaddexp =
R"(logaddexp(x, y)
--

The logarithm of the sum of exponents of x and y. This function is
equivalent to ``log(exp(x) + exp(y))``, but does not suffer from
catastrophic precision loss for small values of x and y.
)";


const char* doc_math_pow =
R"(pow(x, y)
--

Number ``x`` raised to the power ``y``. The return value will be float, even
if the arguments ``x`` and ``y`` are integers.

This function is equivalent to ``x ** y``.
)";


const char* doc_math_rad2deg =
R"(rad2deg(x)
--

Convert angle measured in radians into degrees:
:math:`\operatorname{rad2deg}(x) = x\cdot\frac{360}{\tau}`.

See also
--------
- :func:`deg2rad(x)` -- convert degrees into radians.
)";


const char* doc_math_rint =
R"(rint(x)
--

Round the value `x` to the nearest integer.
)";


const char* doc_math_round =
R"(round(cols, ndigits=None)
--

.. x-version-added:: 0.11

Round the values in `cols` up to the specified number of the digits
of precision `ndigits`. If the number of digits is omitted, rounds
to the nearest integer.

Generally, this operation is equivalent to::

    >>> rint(col * 10**ndigits) / 10**ndigits

where function `rint()` rounds to the nearest integer.

Parameters
----------
cols: FExpr
    Input data for rounding. This could be an expression yielding
    either a single or multiple columns. The `round()` function will
    apply to each column independently and produce as many columns
    in the output as there were in the input.

    Only numeric columns are allowed: boolean, integer or float.
    An exception will be raised if `cols` contains a non-numeric
    column.

ndigits: int | None
    The number of precision digits to retain. This parameter could
    be either positive or negative (or None). If positive then it
    gives the number of digits after the decimal point. If negative,
    then it rounds the result up to the corresponding power of 10.

    For example, `123.45` rounded to `ndigits=1` is `123.4`, whereas
    rounded to `ndigits=-1` it becomes `120.0`.

return: FExpr
    f-expression that rounds the values in its first argument to
    the specified number of precision digits.

    Each input column will produce the column of the same stype in
    the output; except for the case when `ndigits` is `None` and
    the input is either `float32` or `float64`, in which case an
    `int64` column is produced (similarly to python's `round()`).

Notes
-----
Values that are exactly half way in between their rounded neighbors
are converted towards their nearest even value. For example, both
`7.5` and `8.5` are rounded into `8`, whereas `6.5` is rounded as `6`.

Rounding integer columns may produce unexpected results for values
that are close to the min/max value of that column's storage type.
For example, when an `int8` value `127` is rounded to nearest `10`, it
becomes `130`. However, since `130` cannot be represented as `int8` a
wrap-around occurs and the result becomes `-126`.

Rounding an integer column to a positive `ndigits` is a noop: the
column will be returned unchanged.

Rounding an integer column to a large negative `ndigits` will produce
a constant all-0 column.
)";


const char* doc_math_sign =
R"(sign(x)
--

The sign of x, returned as float.

This function returns `1.0` if `x` is positive (including positive
infinity), `-1.0` if `x` is negative, `0.0` if `x` is zero, and NA if
`x` is NA.
)";


const char* doc_math_signbit =
R"(signbit(x)
--

Returns `True` if `x` is negative (its sign bit is set), and `False` if
`x` is positive. This function is able to distinguish between `-0.0` and
`+0.0`, returning `True`/`False` respectively. If `x` is an NA value, this
function will also return NA.
)";


const char* doc_math_sin =
R"(sin(x)
--

Compute the trigonometric sine of angle ``x`` measured in radians.

This function can only be applied to numeric columns (real, integer, or
boolean), and produces a float64 result, except when the argument ``x`` is
float32, in which case the result is float32 as well.

See also
--------
- :func:`cos(x)` -- the trigonometric cosine function;
- :func:`arcsin(x)` -- the inverse sine function.
)";


const char* doc_math_sinh =
R"(sinh(x)
--

Hyperbolic sine of `x`, defined as
:math:`\sinh x = \frac12(e^x - e^{-x})`.

See also
--------
- :func:`cosh` -- hyperbolic cosine;
- :func:`arsinh` -- inverse hyperbolic sine.
)";


const char* doc_math_sqrt =
R"(sqrt(x)
--

The square root of x, same as ``x ** 0.5``.
)";


const char* doc_math_square =
R"(square(x)
--

The square of x, same as ``x ** 2.0``. As with all other math
functions, the result is floating-point, even if the argument
x is integer.
)";


const char* doc_math_tan =
R"(tan(x)
--

Compute the trigonometric tangent of ``x``, which is the ratio
``sin(x)/cos(x)``.

This function can only be applied to numeric columns (real, integer, or
boolean), and produces a float64 result, except when the argument ``x`` is
float32, in which case the result is float32 as well.

See also
--------
- :func:`arctan(x)` -- the inverse tangent function.
)";


const char* doc_math_tanh =
R"(tanh(x)
--

Hyperbolic tangent of `x`, defined as
:math:`\tanh x = \frac{\sinh x}{\cosh x} = \frac{e^x-e^{-x}}{e^x+e^{-x}}`.

See also
--------
- :func:`artanh` -- inverse hyperbolic tangent.
)";


const char* doc_math_trunc =
R"(trunc(x)
--

The nearest integer value not greater than `x` in magnitude.

If x is integer or boolean, then trunc() will return this value
converted to float64. If x is floating-point, then trunc(x) acts as
floor(x) for positive values of x, and as ceil(x) for negative values
of x. This rounding mode is known as rounding towards zero.
)";


const char* doc_models_aggregate =
R"(aggregate(frame, min_rows=500, n_bins=500, nx_bins=50, ny_bins=50,
--

Aggregate a frame into clusters. Each cluster consists of
a set of members, i.e. a subset of the input frame, and is represented by
an exemplar, i.e. one of the members.

For one- and two-column frames the aggregation is based on
the standard equal-interval binning for numeric columns and
grouping operation for string columns.

In the general case, a parallel one-pass ad hoc algorithm is employed.
It starts with an empty exemplar list and does one pass through the data.
If a partucular observation falls into a bubble with a given radius and
the center being one of the exemplars, it marks this observation as a
member of that exemplar's cluster. If there is no appropriate exemplar found,
the observation is marked as a new exemplar.

If the `fixed_radius` is `None`, the algorithm will start
with the `delta`, that is radius squared, being equal to the machine precision.
When the number of gathered exemplars becomes larger than `nd_max_bins`,
the following procedure is performed:

- find the mean distance between all the gathered exemplars;
- merge all the exemplars that are within the half of this distance;
- adjust `delta` by taking into account the initial bubble radius;
- save the exemplar's merging information for the final processing.

If the `fixed_radius` is set to a valid numeric value, the algorithm
will stick to that value and will not adjust `delta`.

**Note:** the general n-dimensional algorithm takes into account the numeric
columns only, and all the other columns are ignored.


Parameters
----------
frame: Frame
    The input frame containing numeric or string columns.

min_rows: int
    Minimum number of rows the input frame should have to be aggregated.
    If `frame` has less rows than `min_rows`, aggregation
    is bypassed, in the sence that all the input rows become exemplars.

n_bins: int
    Number of bins for 1D aggregation.

nx_bins: int
    Number of bins for the first column for 2D aggregation.

ny_bins: int
    Number of bins for the second column for 2D aggregation.

nd_max_bins: int
    Maximum number of exemplars for ND aggregation. It is guaranteed
    that the ND algorithm will return less than `nd_max_bins` exemplars,
    but the exact number may vary from run to run due to parallelization.

max_dimensions: int
    Number of columns at which the projection method is used for
    ND aggregation.

seed: int
    Seed to be used for the projection method.

double_precision: bool
    An option to indicate whether double precision, i.e. `float64`,
    or single precision, i.e. `float32`, arithmetic should be used
    for computations.

fixed_radius: float
    Fixed radius for ND aggregation, use it with caution.
    If set, `nd_max_bins` will have no effect and in the worst
    case number of exemplars may be equal to the number of rows
    in the data. For big data this may result in extremly large
    execution times. Since all the columns are normalized to `[0, 1)`,
    the `fixed_radius` value should be chosen accordingly.

return: Tuple[Frame, Frame]
    The first element in the tuple is the aggregated frame, i.e.
    the frame containing exemplars, with the shape of
    `(nexemplars, frame.ncols + 1)`, where `nexemplars` is
    the number of gathered exemplars. The first `frame.ncols` columns
    are the columns from the input frame, and the last column
    is the `members_count` that has stype `int32` containing
    number of members per exemplar.

    The second element in the tuple is the members frame with the shape of
    `(frame.nrows, 1)`. Each row in this frame corresponds to the
    row with the same id in the input `frame`. The single column `exemplar_id`
    has an stype of `int32` and contains the exemplar ids that a particular
    member belongs to. These ids are effectively the ids of
    the exemplar's rows from the input frame.


except: TypeError
    The exception is raised when one of the `frame`'s columns has an
    unsupported stype, i.e. there is a column that is both non-numeric
    and non-string.
)";


const char* doc_models_kfold =
R"(kfold(nrows, nsplits)
--

Perform k-fold split of data with `nrows` rows into `nsplits` train/test
subsets. The dataset itself is not passed to this function:
it is sufficient to know only the number of rows in order to decide
how the data should be split.

The range `[0; nrows)` is split into `nsplits` approximately equal parts,
i.e. folds, and then each `i`-th split will use the `i`-th fold as a
test part, and all the remaining rows as the train part. Thus, `i`-th split is
comprised of:

  - train rows: `[0; i*nrows/nsplits) + [(i+1)*nrows/nsplits; nrows)`;
  - test rows: `[i*nrows/nsplits; (i+1)*nrows/nsplits)`.

where integer division is assumed.

Parameters
----------
nrows: int
    The number of rows in the frame that is going to be split.

nsplits: int
    Number of folds, must be at least `2`, but not larger than `nrows`.

return: List[Tuple]
    This function returns a list of `nsplits` tuples `(train_rows, test_rows)`,
    where each component of the tuple is a rows selector that can be applied
    to any frame with `nrows` rows to select the desired folds.
    Some of these row selectors will be simple python ranges,
    others will be single-column Frame objects.

See Also
--------
:func:`kfold_random()` -- Perform randomized k-fold split.
)";


const char* doc_models_kfold_random =
R"(kfold_random(nrows, nsplits, seed=None)
--

Perform randomized k-fold split of data with `nrows` rows into
`nsplits` train/test subsets. The dataset itself is not passed to this
function: it is sufficient to know only the number of rows in order to decide
how the data should be split.

The train/test subsets produced by this function will have the following
properties:

  - all test folds will be of approximately the same size `nrows/nsplits`;
  - all observations have equal ex-ante chance of getting assigned into each fold;
  - the row indices in all train and test folds will be sorted.

The function uses single-pass parallelized algorithm to construct the
folds.

Parameters
----------
nrows: int
    The number of rows in the frame that you want to split.

nsplits: int
    Number of folds, must be at least `2`, but not larger than `nrows`.

seed: int
    Seed value for the random number generator used by this function.
    Calling the function several times with the same seed values
    will produce same results each time.

return: List[Tuple]
    This function returns a list of `nsplits` tuples `(train_rows, test_rows)`,
    where each component of the tuple is a rows selector that can be applied to
    to any frame with `nrows` rows to select the desired folds.


See Also
--------
:func:`kfold()` -- Perform k-fold split.
)";


const char* doc_models_Ftrl =
R"(This class implements the Follow the Regularized Leader (FTRL) model,
that is based on the
`FTRL-Proximal <https://www.eecs.tufts.edu/~dsculley/papers/ad-click-prediction.pdf>`_
online learning algorithm for binomial logistic regression. Multinomial
classification and regression for continuous targets are also implemented,
though these implementations are experimental. This model is fully parallel
and is based on the
`Hogwild approach <https://people.eecs.berkeley.edu/~brecht/papers/hogwildTR.pdf>`_
for parallelization.

The model supports numerical (boolean, integer and float types),
temporal (date and time types) and string features. To vectorize features a hashing trick
is employed, such that all the values are hashed with the 64-bit hashing function.
This function is implemented as follows:

- for booleans and integers the hashing function is essentially an identity
  function;

- for floats the hashing function trims mantissa, taking into account
  :attr:`mantissa_nbits <datatable.models.Ftrl.mantissa_nbits>`,
  and interprets the resulting bit representation as a 64-bit unsigned integer;

- for date and time types the hashing function is essentially an identity
  function that is based on their internal integer representations;

- for strings the 64-bit `Murmur2 <https://github.com/aappleby/smhasher>`_
  hashing function is used.

To compute the final hash `x` the Murmur2 hashed feature name is added
to the hashed feature and the result is modulo divided by the number of
requested bins, i.e. by :attr:`nbins <datatable.models.Ftrl.nbins>`.

For each hashed row of data, according to
`Ad Click Prediction: a View from the Trenches <https://www.eecs.tufts.edu/~dsculley/papers/ad-click-prediction.pdf>`_,
the following FTRL-Proximal algorithm is employed:

.. raw:: html

      <img src="../../_static/ftrl_algorithm.png" width="400"
       alt="Per-coordinate FTRL-Proximal online learning algorithm" />

When trained, the model can be used to make predictions, or it can be
re-trained on new datasets as many times as needed improving
model weights from run to run.


Construction
------------

.. list-table::
    :widths: auto
    :class: api-table

    * - :meth:`Ftrl() <datatable.models.Ftrl.__init__>`
      - Construct an `Ftrl` object.



Methods
-------

.. list-table::
    :widths: auto
    :class: api-table

    * - :meth:`fit() <datatable.models.Ftrl.fit>`
      - Train model on the input samples and targets.

    * - :meth:`predict() <datatable.models.Ftrl.predict>`
      - Predict for the input samples.

    * - :meth:`reset() <datatable.models.Ftrl.reset>`
      - Reset the model.



Properties
----------

.. list-table::
    :widths: auto
    :class: api-table

    * - :attr:`alpha <datatable.models.Ftrl.alpha>`
      - :math:`\alpha` in per-coordinate FTRL-Proximal algorithm.

    * - :attr:`beta <datatable.models.Ftrl.beta>`
      - :math:`\beta` in per-coordinate FTRL-Proximal algorithm.

    * - :attr:`colnames <datatable.models.Ftrl.colnames>`
      - Column names of the training frame, i.e. features.

    * - :attr:`colname_hashes <datatable.models.Ftrl.colname_hashes>`
      - Hashes of the column names.

    * - :attr:`double_precision <datatable.models.Ftrl.double_precision>`
      - An option to control precision of the internal computations.

    * - :attr:`feature_importances <datatable.models.Ftrl.feature_importances>`
      - Feature importances calculated during training.

    * - :attr:`interactions <datatable.models.Ftrl.interactions>`
      - Feature interactions.

    * - :attr:`labels <datatable.models.Ftrl.labels>`
      - Classification labels.

    * - :attr:`lambda1 <datatable.models.Ftrl.lambda2>`
      -  L1 regularization parameter, :math:`\lambda_1` in per-coordinate FTRL-Proximal algorithm.

    * - :attr:`lambda2 <datatable.models.Ftrl.lambda2>`
      -  L2 regularization parameter, :math:`\lambda_2` in per-coordinate FTRL-Proximal algorithm.

    * - :attr:`mantissa_nbits <datatable.models.Ftrl.mantissa_nbits>`
      - Number of mantissa bits for hashing floats.

    * - :attr:`model <datatable.models.Ftrl.model>`
      - The model's `z` and `n` coefficients.

    * - :attr:`model_type <datatable.models.Ftrl.model_type>`
      - A model type `Ftrl` should build.

    * - :attr:`model_type_trained <datatable.models.Ftrl.model_type_trained>`
      - A model type `Ftrl` has built.

    * - :attr:`nbins <datatable.models.Ftrl.nbins>`
      - Number of bins for the hashing trick.

    * - :attr:`negative_class <datatable.models.Ftrl.negative_class>`
      - An option to indicate if the "negative" class should be a created
        for multinomial classification.

    * - :attr:`nepochs <datatable.models.Ftrl.nepochs>`
      - Number of training epochs.

    * - :attr:`params <datatable.models.Ftrl.params>`
      - All the input model parameters as a named tuple.
)";


const char* doc_models_Ftrl___init__ =
R"(__init__(self, alpha=0.005, beta=1, lambda1=0, lambda2=0, nbins=10**6,
--

Create a new :class:`Ftrl <datatable.models.Ftrl>` object.

Parameters
----------
alpha: float
    :math:`\alpha` in per-coordinate FTRL-Proximal algorithm, should be
    positive.

beta: float
    :math:`\beta` in per-coordinate FTRL-Proximal algorithm, should be non-negative.

lambda1: float
    L1 regularization parameter, :math:`\lambda_1` in per-coordinate
    FTRL-Proximal algorithm. It should be non-negative.

lambda2: float
    L2 regularization parameter, :math:`\lambda_2` in per-coordinate
    FTRL-Proximal algorithm. It should be non-negative.

nbins: int
    Number of bins to be used for the hashing trick, should be positive.

mantissa_nbits: int
    Number of mantissa bits to take into account when hashing floats.
    It should be non-negative and less than or equal to `52`, that
    is a number of mantissa bits allocated for a C++ 64-bit `double`.

nepochs: float
    Number of training epochs, should be non-negative. When `nepochs`
    is an integer number, the model will train on all the data
    provided to :meth:`.fit` method `nepochs` times. If `nepochs`
    has a fractional part `{nepochs}`, the model will train on all
    the data `[nepochs]` times, i.e. the integer part of `nepochs`.
    Plus, it will also perform an additional training iteration
    on the `{nepochs}` fraction of data.

double_precision: bool
    An option to indicate whether double precision, i.e. `float64`,
    or single precision, i.e. `float32`, arithmetic should be used
    for computations. It is not guaranteed that setting
    `double_precision` to `True` will automatically improve
    the model accuracy. It will, however, roughly double the memory
    footprint of the `Ftrl` object.

negative_class: bool
    An option to indicate if a "negative" class should be created
    in the case of multinomial classification. For the "negative"
    class the model will train on all the negatives, and if
    a new label is encountered in the target column, its
    weights will be initialized to the current "negative" class weights.
    If `negative_class` is set to `False`, the initial weights
    become zeros.

interactions: List[List[str] | Tuple[str]] | Tuple[List[str] | Tuple[str]]
    A list or a tuple of interactions. In turn, each interaction
    should be a list or a tuple of feature names, where each feature
    name is a column name from the training frame. Each interaction
    should have at least one feature.

model_type: "binomial" | "multinomial" | "regression" | "auto"
    The model type to be built. When this option is `"auto"`
    then the model type will be automatically chosen based on
    the target column `stype`.

params: FtrlParams
    Named tuple of the above parameters. One can pass either this tuple,
    or any combination of the individual parameters to the constructor,
    but not both at the same time.

except: ValueError
    The exception is raised if both the `params` and one of the
    individual model parameters are passed at the same time.
)";


const char* doc_models_Ftrl_alpha =
R"(:math:`\alpha` in per-coordinate FTRL-Proximal algorithm.

Parameters
----------
return: float
    Current `alpha` value.

new_alpha: float
    New `alpha` value, should be positive.

except: ValueError
    The exception is raised when `new_alpha` is not positive.
)";


const char* doc_models_Ftrl_beta =
R"(:math:`\beta` in per-coordinate FTRL-Proximal algorithm.

Parameters
----------
return: float
    Current `beta` value.

new_beta: float
    New `beta` value, should be non-negative.

except: ValueError
    The exception is raised when `new_beta` is negative.
)";


const char* doc_models_Ftrl_colname_hashes =
R"(Hashes of the column names used for the hashing trick as
described in the :class:`Ftrl <dt.models.Ftrl>` class description.

Parameters
----------
return: List[int]
    A list of the column name hashes.

See also
--------
- :attr:`.colnames` -- the column names of the
  training frame, i.e. the feature names.
)";


const char* doc_models_Ftrl_colnames =
R"(Column names of the training frame, i.e. the feature names.

Parameters
----------
return: List[str]
    A list of the column names.

See also
--------
- :attr:`.colname_hashes` -- the hashed column names.
)";


const char* doc_models_Ftrl_double_precision =
R"(An option to indicate whether double precision, i.e. `float64`,
or single precision, i.e. `float32`, arithmetic should be
used for computations. This option is read-only and can only be set
during the `Ftrl` object :meth:`construction <datatable.models.Ftrl.__init__>`.

Parameters
----------
return: bool
    Current `double_precision` value.
)";


const char* doc_models_Ftrl_feature_importances =
R"(Feature importances as calculated during the model training and
normalized to `[0; 1]`. The normalization is done by dividing
the accumulated feature importances over the maximum value.

Parameters
----------
return: Frame
    A frame with two columns: `feature_name` that has stype `str32`,
    and `feature_importance` that has stype `float32` or `float64`
    depending on whether the :attr:`.double_precision`
    option is `False` or `True`.
)";


const char* doc_models_Ftrl_fit =
R"(fit(self, X_train, y_train, X_validation=None, y_validation=None,
--

Train model on the input samples and targets.

Parameters
----------
X_train: Frame
    Training frame.

y_train: Frame
    Target frame having as many rows as `X_train` and one column.

X_validation: Frame
    Validation frame having the same number of columns as `X_train`.

y_validation: Frame
    Validation target frame of shape `(nrows, 1)`.

nepochs_validation: float
    Parameter that specifies how often, in epoch units, validation
    error should be checked.

validation_error: float
    The improvement of the relative validation error that should be
    demonstrated by the model within `nepochs_validation` epochs,
    otherwise the training will stop.

validation_average_niterations: int
    Number of iterations that is used to average the validation error.
    Each iteration corresponds to `nepochs_validation` epochs.

return: FtrlFitOutput
    `FtrlFitOutput` is a `Tuple[float, float]` with two fields: `epoch` and `loss`,
    representing the final fitting epoch and the final loss, respectively.
    If validation dataset is not provided, the returned `epoch` equals to
    `nepochs` and the `loss` is just `float('nan')`.

See also
--------
- :meth:`.predict` -- predict for the input samples.
- :meth:`.reset` -- reset the model.
)";


const char* doc_models_Ftrl_interactions =
R"(The feature interactions to be used for model training. This option is
read-only for a trained model.

Parameters
----------
return: Tuple
    Current `interactions` value.

new_interactions: List[List[str] | Tuple[str]] | Tuple[List[str] | Tuple[str]]
    New `interactions` value. Each particular interaction
    should be a list or a tuple of feature names, where each feature
    name is a column name from the training frame.

except: ValueError
    The exception is raised when

    - trying to change this option for a model that has already been trained;
    - one of the interactions has zero features.
)";


const char* doc_models_Ftrl_labels =
R"(Classification labels the model was trained on.

Parameters
----------
return: Frame
    A one-column frame with the classification labels.
    In the case of numeric regression, the label is
    the target column name.
)";


const char* doc_models_Ftrl_lambda1 =
R"(L1 regularization parameter, :math:`\lambda_1` in per-coordinate
FTRL-Proximal algorithm.

Parameters
----------
return: float
    Current `lambda1` value.

new_lambda1: float
    New `lambda1` value, should be non-negative.

except: ValueError
    The exception is raised when `new_lambda1` is negative.
)";


const char* doc_models_Ftrl_lambda2 =
R"(L2 regularization parameter, :math:`\lambda_2` in per-coordinate
FTRL-Proximal algorithm.

Parameters
----------
return: float
    Current `lambda2` value.

new_lambda2: float
    New `lambda2` value, should be non-negative.

except: ValueError
    The exception is raised when `new_lambda2` is negative.
)";


const char* doc_models_Ftrl_mantissa_nbits =
R"(Number of mantissa bits to take into account for hashing floats.
This option is read-only for a trained model.

Parameters
----------
return: int
    Current `mantissa_nbits` value.

new_mantissa_nbits: int
    New `mantissa_nbits` value, should be non-negative and
    less than or equal to `52`, that is a number of
    mantissa bits in a C++ 64-bit `double`.

except: ValueError
    The exception is raised when

    - trying to change this option for a model that has already been trained;
    - `new_mantissa_nbits` value is negative or larger than `52`.
)";


const char* doc_models_Ftrl_model =
R"(Trained models weights, i.e. `z` and `n` coefficients
in per-coordinate FTRL-Proximal algorithm.

Parameters
----------
return: Frame
    A frame of shape `(nbins, 2 * nlabels)`, where `nlabels` is
    the total number of labels the model was trained on, and
    :attr:`nbins <datatable.models.Ftrl.nbins>` is the number of bins
    used for the hashing trick. Odd and even columns represent
    the `z` and `n` model coefficients, respectively.
)";


const char* doc_models_Ftrl_model_type =
R"(A type of the model `Ftrl` should build:

- `"binomial"` for binomial classification;
- `"multinomial"` for multinomial classification;
- `"regression"` for numeric regression;
- `"auto"` for automatic model type detection based on the target column `stype`.

This option is read-only for a trained model.

Parameters
----------
return: str
    Current `model_type` value.

new_model_type: "binomial" | "multinomial" | "regression" | "auto"
    New `model_type` value.

except: ValueError
    The exception is raised when

    - trying to change this option for a model that has already been trained;
    - `new_model_type` value is not one of the following: `"binomial"`,
      `"multinomial"`, `"regression"` or `"auto"`.

See also
--------
- :attr:`.model_type_trained` -- the model type `Ftrl` has build.
)";


const char* doc_models_Ftrl_model_type_trained =
R"(The model type `Ftrl` has built.

Parameters
----------
return: str
    Could be one of the following: `"regression"`, `"binomial"`,
    `"multinomial"` or `"none"` for untrained model.

See also
--------
- :attr:`.model_type` -- the model type `Ftrl` should build.
)";


const char* doc_models_Ftrl_nbins =
R"(Number of bins to be used for the hashing trick.
This option is read-only for a trained model.

Parameters
----------
return: int
    Current `nbins` value.

new_nbins: int
    New `nbins` value, should be positive.

except: ValueError
    The exception is raised when

    - trying to change this option for a model that has already been trained;
    - `new_nbins` value is not positive.
)";


const char* doc_models_Ftrl_negative_class =
R"(An option to indicate if a "negative" class should be created
in the case of multinomial classification. For the "negative"
class the model will train on all the negatives, and if
a new label is encountered in the target column, its
weights are initialized to the current "negative" class weights.
If `negative_class` is set to `False`, the initial weights
become zeros.

This option is read-only for a trained model.

Parameters
----------
return: bool
    Current `negative_class` value.

new_negative_class: bool
    New `negative_class` value.

except: ValueError
    The exception is raised when trying to change this option
    for a model that has already been trained.
)";


const char* doc_models_Ftrl_nepochs =
R"(Number of training epochs. When `nepochs` is an integer number,
the model will train on all the data provided to :meth:`.fit` method
`nepochs` times. If `nepochs` has a fractional part `{nepochs}`,
the model will train on all the data `[nepochs]` times,
i.e. the integer part of `nepochs`. Plus, it will also perform an additional
training iteration on the `{nepochs}` fraction of data.

Parameters
----------
return: float
    Current `nepochs` value.

new_nepochs: float
    New `nepochs` value, should be non-negative.

except: ValueError
    The exception is raised when `new_nepochs` value is negative.
)";


const char* doc_models_Ftrl_params =
R"(`Ftrl` model parameters as a named tuple `FtrlParams`,
see :meth:`.__init__` for more details.
This option is read-only for a trained model.

Parameters
----------
return: FtrlParams
    Current `params` value.

new_params: FtrlParams
    New `params` value.

except: ValueError
    The exception is raised when

    - trying to change this option for a model that has already been trained;
    - individual parameter values are incompatible with the corresponding setters.
)";


const char* doc_models_Ftrl_predict =
R"(predict(self, X)
--

Predict for the input samples.

Parameters
----------
X: Frame
    A frame to make predictions for. It should have the same number
    of columns as the training frame.

return: Frame
    A new frame of shape `(X.nrows, nlabels)` with the predicted probabilities
    for each row of frame `X` and each of `nlabels` labels
    the model was trained for.

See also
--------
- :meth:`.fit` -- train model on the input samples and targets.
- :meth:`.reset` -- reset the model.
)";


const char* doc_models_Ftrl_reset =
R"(reset(self)
--

Reset `Ftrl` model by resetting all the model weights, labels and
feature importance information.

Parameters
----------
return: None

See also
--------
- :meth:`.fit` -- train model on a dataset.
- :meth:`.predict` -- predict on a dataset.
)";


const char* doc_models_LinearModel =
R"(This class implements the `Linear model`_ with the
`stochastic gradient descent`_ learning. It supports linear
regression, as well as binomial and multinomial classification.
Both :meth:`.fit` and :meth:`.predict` methods are fully parallel.

.. _`Linear model`: https://en.wikipedia.org/wiki/Linear_model
.. _`stochastic gradient descent`: https://en.wikipedia.org/wiki/Stochastic_gradient_descent


Construction
------------

.. list-table::
    :widths: auto
    :class: api-table

    * - :meth:`LinearModel() <datatable.models.LinearModel.__init__>`
      - Construct a `LinearModel` object.



Methods
-------

.. list-table::
    :widths: auto
    :class: api-table

    * - :meth:`fit() <datatable.models.LinearModel.fit>`
      - Train model on the input samples and targets.

    * - :meth:`is_fitted() <datatable.models.LinearModel.is_fitted>`
      - Report model status.

    * - :meth:`predict() <datatable.models.LinearModel.predict>`
      - Predict for the input samples.

    * - :meth:`reset() <datatable.models.LinearModel.reset>`
      - Reset the model.


Properties
----------

.. list-table::
    :widths: auto
    :class: api-table

    * - :attr:`eta0 <datatable.models.LinearModel.eta0>`
      - Initial learning rate.

    * - :attr:`eta_decay <datatable.models.LinearModel.eta_decay>`
      - Decay for the `"time-based"` and `"step-based"` learning rate schedules.

    * - :attr:`eta_drop_rate <datatable.models.LinearModel.eta_drop_rate>`
      - Drop rate for the `"step-based"` learning rate schedule.

    * - :attr:`eta_schedule <datatable.models.LinearModel.eta_schedule>`
      - Learning rate schedule.

    * - :attr:`double_precision <datatable.models.LinearModel.double_precision>`
      - An option to control precision of the internal computations.

    * - :attr:`labels <datatable.models.LinearModel.labels>`
      - Classification labels.

    * - :attr:`lambda1 <datatable.models.LinearModel.lambda2>`
      - L1 regularization parameter.

    * - :attr:`lambda2 <datatable.models.LinearModel.lambda2>`
      - L2 regularization parameter.

    * - :attr:`model <datatable.models.LinearModel.model>`
      - Model coefficients.

    * - :attr:`model_type <datatable.models.LinearModel.model_type>`
      - Model type to be built.

    * - :attr:`negative_class <datatable.models.LinearModel.negative_class>`
      - An option to indicate if the "negative" class should be a created
        for multinomial classification.

    * - :attr:`nepochs <datatable.models.LinearModel.nepochs>`
      - Number of training epochs.

    * - :attr:`params <datatable.models.LinearModel.params>`
      - All the input model parameters as a named tuple.

    * - :attr:`seed <datatable.models.LinearModel.seed>`
      - Seed for the quasi-random data shuffling.
)";


const char* doc_models_LinearModel___init__ =
R"(__init__(self,
--

Create a new :class:`LinearModel <datatable.models.LinearModel>` object.

Parameters
----------
eta0: float
    The initial learning rate, should be positive.

eta_decay: float
    Decay for the `"time-based"` and `"step-based"`
    learning rate schedules, should be non-negative.

eta_drop_rate: float
    Drop rate for the `"step-based"` learning rate schedule,
    should be positive.

eta_schedule: "constant" | "time-based" | "step-based" | "exponential"
    Learning rate schedule. When it is `"constant"` the learning rate
    `eta` is constant and equals to `eta0`. Otherwise,
    after each training iteration `eta` is updated as follows:

    - for `"time-based"` schedule as `eta0 / (1 + eta_decay * epoch)`;
    - for `"step-based"` schedule as `eta0 * eta_decay ^ floor((1 + epoch) / eta_drop_rate)`;
    - for `"exponential"` schedule as `eta0 / exp(eta_decay * epoch)`.

    By default, the size of the training iteration is one epoch, it becomes
    `nepochs_validation` when validation dataset is specified.

lambda1: float
    L1 regularization parameter, should be non-negative.

lambda2: float
    L2 regularization parameter, should be non-negative.

nepochs: float
    Number of training epochs, should be non-negative. When `nepochs`
    is an integer number, the model will train on all the data
    provided to :meth:`.fit` method `nepochs` times. If `nepochs`
    has a fractional part `{nepochs}`, the model will train on all
    the data `[nepochs]` times, i.e. the integer part of `nepochs`.
    Plus, it will also perform an additional training iteration
    on the `{nepochs}` fraction of data.

double_precision: bool
    An option to indicate whether double precision, i.e. `float64`,
    or single precision, i.e. `float32`, arithmetic should be used
    for computations. It is not guaranteed that setting
    `double_precision` to `True` will automatically improve
    the model accuracy. It will, however, roughly double the memory
    footprint of the `LinearModel` object.

negative_class: bool
    An option to indicate if a "negative" class should be created
    in the case of multinomial classification. For the "negative"
    class the model will train on all the negatives, and if
    a new label is encountered in the target column, its
    coefficients will be initialized to the current "negative" class coefficients.
    If `negative_class` is set to `False`, the initial coefficients
    become zeros.

model_type: "binomial" | "multinomial" | "regression" | "auto"
    The model type to be built. When this option is `"auto"`
    then the model type will be automatically chosen based on
    the target column `stype`.

seed: int
    Seed for the quasi-random number generator that is used for
    data shuffling when fitting the model, should be non-negative.
    If seed is zero, no shuffling is performed.

params: LinearModelParams
    Named tuple of the above parameters. One can pass either this tuple,
    or any combination of the individual parameters to the constructor,
    but not both at the same time.

except: ValueError
    The exception is raised if both the `params` and one of the
    individual model parameters are passed at the same time.
)";


const char* doc_models_LinearModel_double_precision =
R"(An option to indicate whether double precision, i.e. `float64`,
or single precision, i.e. `float32`, arithmetic should be
used for computations. This option is read-only and can only be set
during the `LinearModel` object :meth:`construction <datatable.models.LinearModel.__init__>`.

Parameters
----------
return: bool
    Current `double_precision` value.
)";


const char* doc_models_LinearModel_eta0 =
R"(Learning rate.

Parameters
----------
return: float
    Current `eta0` value.

new_eta0: float
    New `eta0` value, should be positive.

except: ValueError
    The exception is raised when `new_eta0` is not positive.
)";


const char* doc_models_LinearModel_eta_decay =
R"(Decay for the `"time-based"` and `"step-based"` learning rate schedules.

Parameters
----------
return: float
    Current `eta_decay` value.

new_eta_decay: float
    New `eta_decay` value, should be non-negative.

except: ValueError
    The exception is raised when `new_eta_decay` is negative.
)";


const char* doc_models_LinearModel_eta_drop_rate =
R"(Drop rate for the `"step-based"` learning rate schedule.

Parameters
----------
return: float
    Current `eta_drop_rate` value.

new_eta_drop_rate: float
    New `eta_drop_rate` value, should be positive.

except: ValueError
    The exception is raised when `new_eta_drop_rate` is not positive.
)";


const char* doc_models_LinearModel_eta_schedule =
R"(Learning rate `schedule <https://en.wikipedia.org/wiki/Learning_rate#Learning_rate_schedule>`_

- `"constant"` for constant `eta`;
- `"time-based"` for time-based schedule;
- `"step-based"` for step-based schedule;
- `"exponential"` for exponential schedule.

Parameters
----------
return: str
    Current `eta_schedule` value.

new_eta_schedule: "constant" | "time-based" | "step-based" | "exponential"
    New `eta_schedule` value.
)";


const char* doc_models_LinearModel_fit =
R"(fit(self, X_train, y_train, X_validation=None, y_validation=None,
--

Train model on the input samples and targets using the
`parallel stochastic gradient descent <http://martin.zinkevich.org/publications/nips2010.pdf>`_
method.

Parameters
----------
X_train: Frame
    Training frame.

y_train: Frame
    Target frame having as many rows as `X_train` and one column.

X_validation: Frame
    Validation frame having the same number of columns as `X_train`.

y_validation: Frame
    Validation target frame of shape `(nrows, 1)`.

nepochs_validation: float
    Parameter that specifies how often, in epoch units, validation
    error should be checked.

validation_error: float
    The improvement of the relative validation error that should be
    demonstrated by the model within `nepochs_validation` epochs,
    otherwise the training will stop.

validation_average_niterations: int
    Number of iterations that is used to average the validation error.
    Each iteration corresponds to `nepochs_validation` epochs.

return: LinearModelFitOutput
    `LinearModelFitOutput` is a `Tuple[float, float]` with two fields: `epoch` and `loss`,
    representing the final fitting epoch and the final loss, respectively.
    If validation dataset is not provided, the returned `epoch` equals to
    `nepochs` and the `loss` is just `float('nan')`.

See also
--------
- :meth:`.predict` -- predict for the input samples.
- :meth:`.reset` -- reset the model.
)";


const char* doc_models_LinearModel_is_fitted =
R"(is_fitted(self)
--

Report model status.

Parameters
----------
return: bool
    `True` if model is trained, `False` otherwise.
)";


const char* doc_models_LinearModel_labels =
R"(Classification labels the model was trained on.

Parameters
----------
return: Frame
    A one-column frame with the classification labels.
    In the case of numeric regression, the label is
    the target column name.
)";


const char* doc_models_LinearModel_lambda1 =
R"(L1 regularization parameter.

Parameters
----------
return: float
    Current `lambda1` value.

new_lambda1: float
    New `lambda1` value, should be non-negative.

except: ValueError
    The exception is raised when `new_lambda1` is negative.
)";


const char* doc_models_LinearModel_lambda2 =
R"(L2 regularization parameter.

Parameters
----------
return: float
    Current `lambda2` value.

new_lambda2: float
    New `lambda2` value, should be non-negative.

except: ValueError
    The exception is raised when `new_lambda2` is negative.
)";


const char* doc_models_LinearModel_model =
R"(Trained models coefficients.

Parameters
----------
return: Frame
    A frame of shape `(nfeatures + 1, nlabels)`, where `nlabels` is
    the number of labels the model was trained on, and
    `nfeatures` is the number of features. Each column contains
    model coefficients for the corresponding label: starting from
    the intercept and following by the coefficients for each of
    of the `nfeatures` features.
)";


const char* doc_models_LinearModel_model_type =
R"(A type of the model `LinearModel` should build:

- `"binomial"` for binomial classification;
- `"multinomial"` for multinomial classification;
- `"regression"` for numeric regression;
- `"auto"` for automatic model type detection based on the target column `stype`.

This option is read-only for a trained model.

Parameters
----------
return: str
    Current `model_type` value.

new_model_type: "binomial" | "multinomial" | "regression" | "auto"
    New `model_type` value.

except: ValueError
    The exception is raised when trying to change this option
    for a model that has already been trained.
)";


const char* doc_models_LinearModel_negative_class =
R"(An option to indicate if a "negative" class should be created
in the case of multinomial classification. For the "negative"
class the model will train on all the negatives, and if
a new label is encountered in the target column, its
coefficients are initialized to the current "negative" class coefficients.
If `negative_class` is set to `False`, the initial coefficients
become zeros.

This option is read-only for a trained model.

Parameters
----------
return: bool
    Current `negative_class` value.

new_negative_class: bool
    New `negative_class` value.

except: ValueError
    The exception is raised when trying to change this option
    for a model that has already been trained.
)";


const char* doc_models_LinearModel_nepochs =
R"(Number of training epochs. When `nepochs` is an integer number,
the model will train on all the data provided to :meth:`.fit` method
`nepochs` times. If `nepochs` has a fractional part `{nepochs}`,
the model will train on all the data `[nepochs]` times,
i.e. the integer part of `nepochs`. Plus, it will also perform an additional
training iteration on the `{nepochs}` fraction of data.

Parameters
----------
return: float
    Current `nepochs` value.

new_nepochs: float
    New `nepochs` value, should be non-negative.

except: ValueError
    The exception is raised when `new_nepochs` value is negative.
)";


const char* doc_models_LinearModel_params =
R"(`LinearModel` model parameters as a named tuple `LinearModelParams`,
see :meth:`.__init__` for more details.
This option is read-only for a trained model.

Parameters
----------
return: LinearModelParams
    Current `params` value.

new_params: LinearModelParams
    New `params` value.

except: ValueError
    The exception is raised when

    - trying to change this option for a model that has already been trained;
    - individual parameter values are incompatible with the corresponding setters.
)";


const char* doc_models_LinearModel_predict =
R"(predict(self, X)
--

Predict for the input samples.

Parameters
----------
X: Frame
    A frame to make predictions for. It should have the same number
    of columns as the training frame.

return: Frame
    A new frame of shape `(X.nrows, nlabels)` with the predicted probabilities
    for each row of frame `X` and each of `nlabels` labels
    the model was trained for.

See also
--------
- :meth:`.fit` -- train model on the input samples and targets.
- :meth:`.reset` -- reset the model.
)";


const char* doc_models_LinearModel_reset =
R"(reset(self)
--

Reset linear model by resetting all the model coefficients and labels.

Parameters
----------
return: None

See also
--------
- :meth:`.fit` -- train model on a dataset.
- :meth:`.predict` -- predict on a dataset.
)";


const char* doc_models_LinearModel_seed =
R"(Seed for the quasi-random number generator that is used for
data shuffling when fitting the model. If seed is `0`,
no shuffling is performed.

Parameters
----------
return: int
    Current `seed` value.

new_seed: int
    New `seed` value, should be non-negative.
)";


const char* doc_options_debug_arg_max_size =
R"(This option limits the display size of each argument in order
to prevent potentially huge outputs. It has no effect,
if :attr:`debug.report_args <datatable.options.debug.report_args>` is
`False`.

Parameters
----------
return: int
    Current `arg_max_size` value. Initially, this option is set to `100`.

value: int
    New `arg_max_size` value, should be non-negative.
    If `value < 10`, then `arg_max_size` will be set to `10`.

except: TypeError
    The exception is raised when `value` is negative.
)";


const char* doc_options_debug_enabled =
R"(This option controls whether or not all the calls to the datatable core
functions should be logged.


Parameters
----------
return: bool
    Current `enabled` value. Initially, this option is set to `False`.

value: bool
    New `enabled` value. If set to `True`, all the calls to the datatable
    core functions will be logged along with their respective timings.
)";


const char* doc_options_debug_logger =
R"(The logger object used for reporting calls to datatable core
functions. This option has no effect if
:attr:`debug.enabled <datatable.options.debug.enabled>` is `False`.

Parameters
----------
return: object
    Current `logger` value. Initially, this option is set to `None`,
    meaning that the built-in logger should be used.

value: object
    New `logger` value.

except: TypeError
    The exception is raised when `value` is not an object
    having a method `.debug(self, msg)`.
)";


const char* doc_options_debug_report_args =
R"(This option controls whether log messages for the function
and method calls should contain information about the arguments
of those calls.

Parameters
----------
return: bool
    Current `report_args` value. Initially, this option is set to `False`.

value: object
    New `report_args` value.
)";


const char* doc_options_display_allow_unicode =
R"(This option controls whether or not unicode characters are
allowed in the datatable output.

Parameters
----------
return: bool
    Current `allow_unicode` value. Initially, this option is set to `True`.

value: bool
    New `allow_unicode` value. If `True`, datatable will allow unicode
    characters (encoded as UTF-8) to be printed into the output.
    If `False`, then unicode characters will either be avoided, or
    hex-escaped as necessary.
)";


const char* doc_options_display_head_nrows =
R"(This option controls the number of rows from the top of a frame
to be displayed when the frame's output is truncated due to the total number of
rows exceeding :attr:`display.max_nrows <datatable.options.display.max_nrows>`
value.

Parameters
----------
return: int
    Current `head_nrows` value. Initially, this option is set to `15`.

value: int
    New `head_nrows` value, should be non-negative.

except: ValueError
    The exception is raised when the `value` is negative.
)";


const char* doc_options_display_interactive =
R"(..  warning::

    This option is currently not working properly
    `[#2669] <https://github.com/h2oai/datatable/issues/2669>`_

This option controls the behavior of a Frame when it is viewed in a
text console. To enter the interactive mode manually, one can still
call the :meth:`Frame.view() <dt.Frame.view>` method.

Parameters
----------
return: bool
    Current `interactive` value. Initially, this option is set to `False`.

value: bool
    New `interactive` value. If `True`, frames will be shown in
    the interactove mode, allowing you to navigate the rows/columns
    with the keyboard. If `False`, frames will be shown in regular,
    non-interactive mode.
)";


const char* doc_options_display_max_column_width =
R"(This option controls the threshold for the column's width
to be truncated. If a column's name or its values exceed
the `max_column_width`, the content of the column is truncated
to `max_column_width` characters when printed.

This option applies to both the rendering of a frame in a terminal,
and the rendering in a Jupyter notebook. However, it will have no effect
for data in fixed-width columns, such as boolean, numeric, date or time.

Parameters
----------
return: int
    Current `max_column_width` value. Initially, this option is set to `100`.

value: int
    New `max_column_width` value, cannot be less than `2`.
    If `value` equals to `None`, the column's content
    would never be truncated.

except: ValueError
    The exception is raised when the `value` is less than `2`.
)";


const char* doc_options_display_max_nrows =
R"(This option controls the threshold for the number of rows
in a frame to be truncated when printed to the console.

If a frame has more rows than `max_nrows`, it will be displayed
truncated: only its first
:attr:`head_nrows <datatable.options.display.head_nrows>`
and last
:attr:`tail_nrows <datatable.options.display.tail_nrows>`
rows will be printed. Otherwise, no truncation will occur.
It is recommended to have `head_nrows + tail_nrows <= max_nrows`.

Parameters
----------
return: int
    Current `max_nrows` value. Initially, this option is set to `30`.

value: int
    New `max_nrows` value. If this option is set to `None` or
    to a negative value, no frame truncation will occur when printed,
    which may cause the console to become unresponsive for frames
    with large number of rows.
)";


const char* doc_options_display_tail_nrows =
R"(This option controls the number of rows from the bottom of a frame
to be displayed when the frame's output is truncated due to the total number of
rows exceeding :attr:`max_nrows <datatable.options.display.max_nrows>`
value.

Parameters
----------
return: int
    Current `tail_nrows` value. Initially, this option is set to `5`.

value: int
    New `tail_nrows` value, should be non-negative.

except: ValueError
    The exception is raised when the `value` is negative.
)";


const char* doc_options_display_use_colors =
R"(This option controls whether or not to use colors when printing
datatable messages into the console. Turn this off if your terminal is unable to
display ANSI escape sequences, or if the colors make output not legible.

Parameters
----------
return: bool
    Current `use_colors` value. Initially, this option is set to `True`.

value: bool
    New `use_colors` value.
)";


const char* doc_options_frame_names_auto_index =
R"(This option controls the starting index that is used for auto-naming
the columns. By default, the names that datatable assigns to frame's columns are
`C0`, `C1`, `C2`, etc. Setting `names_auto_index`, for instance, to
`1` will cause the columns to be named as `C1`, `C2`, `C3`, etc.

Parameters
----------
return: int
    Current `names_auto_index` value. Initially, this option is set to `0`.

new_names_auto_index: int
    New `names_auto_index` value.

See Also
--------
- :ref:`name-mangling` -- tutorial on name mangling.
)";


const char* doc_options_frame_names_auto_prefix =
R"(This option controls the prefix that is used for auto-naming
the columns. By default, the names that datatable assigns to frame's columns are
`C0`, `C1`, `C2`, etc. Setting `names_auto_prefix`, for instance, to
`Z` will cause the columns to be named as `Z1`, `Z2`, `Z3`, etc.

Parameters
----------
return: str
    Current `names_auto_prefix` value. Initially, this option is set to `C`.

new_names_auto_prefix: str
    New `names_auto_prefix` value.

See Also
--------
- :ref:`name-mangling` -- tutorial on name mangling.
)";


const char* doc_options_fread_log_anonymize =
R"(This option controls logs anonymization that is useful in production
systems, when reading sensitive data that must not accidentally leak
into log files or be printed with the error messages.

Parameters
----------
return: bool
    Current `anonymize` value. Initially, this option is set to `False`.

value: bool
    New `anonymize` value. If `True`, any snippets of data being read
    that are printed in the log will be first anonymized by converting
    all non-zero digits to `1`, all lowercase letters to `a`,
    all uppercase letters to `A`, and all unicode characters to `U`.
    If `False`, no data anonymization will be performed.
)";


const char* doc_options_fread_log_escape_unicode =
R"(This option controls escaping of the unicode characters.

Use this option if your terminal cannot print unicode,
or if the output gets somehow corrupted because of the unicode characters.

Parameters
----------
return: bool
    Current `escape_unicode` value. Initially, this option is set to `False`.

value: bool
    If `True`, all unicode characters in the verbose log will be written
    in hexadecimal notation. If `False`, no escaping of the unicode
    characters will be performed.
)";


const char* doc_options_fread_parse_dates =
R"(If True, fread will attempt to detect columns of date32 type. If False,
then columns with date values will be returned as strings.

.. warning::

    This option is temporary and will be removed in the future.
)";


const char* doc_options_fread_parse_times =
R"(If True, fread will attempt to detect columns of time64 type. If False,
then columns with timestamps will be returned as strings.

.. warning::

    This option is temporary and will be removed in the future.
)";


const char* doc_options_nthreads =
R"(This option controls the number of threads used by datatable
for parallel calculations.

Parameters
----------
return: int
    Current `nthreads` value. Initially, this option is set to
    the value returned by C++ call `std::thread::hardware_concurrency()`,
    and usually equals to the number of available cores.

new_nthreads: int
    New `nthreads` value. It can be greater or smaller than the initial setting.
    For example, setting `nthreads = 1` will force the library into
    a single-threaded mode. Setting `nthreads` to `0` will restore
    the initial value equal to the number of processor cores.
    Setting `nthreads` to a value less than `0` is equivalent to requesting
    that fewer threads than the maximum.
)";


const char* doc_options_progress_allow_interruption =
R"(This option controls if the datatable tasks could be interrupted.


Parameters
----------
return: bool
    Current `allow_interruption` value. Initially, this option is set to `True`.

value: bool
    New `allow_interruption` value. If `True`, datatable will be allowed
    to handle the `SIGINT` signal to interrupt long-running tasks.
    If `False`, it will not be possible to interrupt tasks with `SIGINT`.
)";


const char* doc_options_progress_callback =
R"(This option controls the custom progress-reporting function.


Parameters
----------
return: function
    Current `callback` value. Initially, this option is set to `None`.

value: function
    New `callback` value. If `None`, then the built-in progress-reporting
    function will be used. Otherwise, the `value` specifies a function
    to be called at each progress event. The function should take a single
    parameter `p`, which is a namedtuple with the following fields:

    - `p.progress` is a float in the range `0.0 .. 1.0`;
    - `p.status` is a string, one of `'running'`, `'finished'`, `'error'` or
      `'cancelled'`;
    - `p.message` is a custom string describing the operation currently
      being performed.
)";


const char* doc_options_progress_clear_on_success =
R"(This option controls if the progress bar is cleared on success.


Parameters
----------
return: bool
    Current `clear_on_success` value. Initially, this option is set to `False`.

value: bool
    New `clear_on_success` value. If `True`, the progress bar is cleared when
    job finished successfully. If `False`, the progress remains visible
    even when the job has already finished.
)";


const char* doc_options_progress_enabled =
R"(This option controls if the progress reporting is enabled.

Parameters
----------
return: bool
    Current `enabled` value. Initially, this option is set to `True`
    if the `stdout` is connected to a terminal or a Jupyter Notebook,
    and `False` otherwise.

value: bool
    New `enabled` value. If `True`, the progress reporting
    functionality will be turned on. If `False`, it is turned off.
)";


const char* doc_options_progress_min_duration =
R"(This option controls the minimum duration of a task to show the progress bar.


Parameters
----------
return: float
    Current `min_duration` value. Initially, this option is set to `0.5`.

value: float
    New `min_duration` value. The progress bar will not be shown
    if the duration of an operation is smaller than `value`.
    If this value is non-zero, then the progress bar will only be shown
    for long-running operations, whose duration (estimated or actual)
    exceeds this threshold.
)";


const char* doc_options_progress_updates_per_second =
R"(This option controls the progress bar update frequency.


Parameters
----------
return: float
    Current `updates_per_second` value. Initially, this option is set to `25.0`.

value: float
    New `updates_per_second` value. This is the number of times per second
    the display of the progress bar should be updated.
)";


const char* doc_re_match =
R"(match(column, pattern, icase=False)
--

Test whether values in a string column match a regular expression.


Parameters
----------
column: FExpr[str]
    The column expression where you want to search for regular
    expression matches.

pattern: str
    The regular expression that will be tested against each value
    in the `column`.


icase: bool

    .. x-version-added:: 1.1.0

    If `True`, character matching will be performed without regard to case.

return: FExpr[bool8]
    A boolean column that tells whether the value in each row of
    `column` matches the `pattern` or not.
)";


const char* doc_str_len =
R"(len(column)
--

Compute lengths of values in a string column.


Parameters
----------
column: FExpr[str]

return: FExpr[int64]
)";


const char* doc_str_slice =
R"(slice(col, start, stop, step=1)
--

Apply slice ``[start:stop:step]`` to each value in a `column` of string
type.

Instead of this function you can directly apply a slice expression to
the column expression::

    - ``f.A[1:-1]`` is equivalent to
    - ``dt.str.slice(f.A, 1, -1)``.


Parameters
----------
col : FExpr[str]
    The column to which the slice should be applied.

return: FExpr[str]
    A column containing sliced string values from the source column.


Examples
--------
>>> DT = dt.Frame(A=["apples", "bananas", "cherries", "dates",
...                  "eggplants", "figs", "grapes", "kiwi"])
>>> DT[:, dt.str.slice(f.A, None, 5)]
   | A
   | str32
-- + -----
 0 | apple
 1 | banan
 2 | cherr
 3 | dates
 4 | eggpl
 5 | figs
 6 | grape
 7 | kiwi
[8 rows x 1 column]
)";


const char* doc_str_split_into_nhot =
R"(split_into_nhot(frame, sep=",", sort=False)
--

Split and nhot-encode a single-column frame.

Each value in the frame, having a single string column, is split according
to the provided separator `sep`, the whitespace is trimmed, and
the resulting pieces (labels) are converted into the individual columns
of the output frame.


Parameters
----------
frame: Frame
    An input single-column frame. The column stype must be either `str32`
    or `str64`.

sep: str
    Single-character separator to be used for splitting.

sort: bool
    An option to control whether the resulting column names, i.e. labels,
    should be sorted. If set to `True`, the column names are returned in
    alphabetical order, otherwise their order is not guaranteed
    due to the algorithm parallelization.

return: Frame
    The output frame. It will have as many rows as the input frame,
    and as many boolean columns as there were unique labels found.
    The labels will also become the output column names.

except: ValueError | TypeError
    :exc:`dt.exceptions.ValueError`
        Raised if the input frame is missing or it has more
        than one column. It is also raised if `sep` is not a single-character
        string.

    :exc:`dt.exceptions.TypeError`
        Raised if the single column of `frame` has non-string stype.


Examples
--------

.. code-block:: python

    >>> DT = dt.Frame(["cat,dog", "mouse", "cat,mouse", "dog,rooster", "mouse,dog,cat"])
    >>> DT
       | C0
       | str32
    -- + -------------
     0 | cat,dog
     1 | mouse
     2 | cat,mouse
     3 | dog,rooster
     4 | mouse,dog,cat
    [5 rows x 1 column]

    >>> dt.split_into_nhot(DT)
       |   cat    dog  mouse  rooster
       | bool8  bool8  bool8    bool8
    -- + -----  -----  -----  -------
     0 |     1      1      0        0
     1 |     0      0      1        0
     2 |     1      0      1        0
     3 |     0      1      0        1
     4 |     1      1      1        0
    [5 rows x 4 columns]
)";


const char* doc_time_day =
R"(day(date)
--

.. x-version-added:: 1.0.0

Retrieve the "day" component of a date32 or time64 column.


Parameters
----------
date: FExpr[date32] | FExpr[time64]
    A column for which you want to compute the day part.

return: FExpr[int32]
    The day part of the source column.


Examples
--------
>>> DT = dt.Frame([1, 1000, 100000], stype='date32')
>>> DT[:, {'date': f[0], 'day': dt.time.day(f[0])}]
   | date          day
   | date32      int32
-- + ----------  -----
 0 | 1970-01-02      2
 1 | 1972-09-27     27
 2 | 2243-10-17     17
[3 rows x 2 columns]


See Also
--------
- :func:`year()` -- retrieve the "year" component of a date
- :func:`month()` -- retrieve the "month" component of a date
)";


const char* doc_time_day_of_week =
R"(day_of_week(date)
--

.. x-version-added:: 1.0.0

For a given date column compute the corresponding days of week.

Days of week are returned as integers from 1 to 7, where 1 represents
Monday, and 7 is Sunday. Thus, the return value of this function
matches the ISO standard.


Parameters
----------
date: FExpr[date32] | FExpr[time64]
    The date32 (or time64) column for which you need to calculate days of week.

return: FExpr[int32]
    An integer column, with values between 1 and 7 inclusive.


Examples
--------
>>> DT = dt.Frame([18000, 18600, 18700, 18800, None], stype='date32')
>>> DT[:, {"date": f[0], "day-of-week": dt.time.day_of_week(f[0])}]
   | date        day-of-week
   | date32            int32
-- + ----------  -----------
 0 | 2019-04-14            7
 1 | 2020-12-04            5
 2 | 2021-03-14            7
 3 | 2021-06-22            2
 4 | NA                   NA
[5 rows x 2 columns]
)";


const char* doc_time_hour =
R"(hour(time)
--

.. x-version-added:: 1.0.0

Retrieve the "hour" component of a time64 column. The returned value
will always be in the range [0; 23].


Parameters
----------
time: FExpr[time64]
    A column for which you want to compute the hour part.

return: FExpr[int32]
    The hour part of the source column.


Examples
--------
>>> from datetime import datetime as d
>>> DT = dt.Frame([d(2020, 5, 11, 12, 0, 0), d(2021, 6, 14, 16, 10, 59, 394873)])
>>> DT[:, {'time': f[0], 'hour': dt.time.hour(f[0])}]
   | time                         hour
   | time64                      int32
-- + --------------------------  -----
 0 | 2020-05-11T12:00:00            12
 1 | 2021-06-14T16:10:59.394873     16
[2 rows x 2 columns]


See Also
--------
- :func:`minute()` -- retrieve the "minute" component of a timestamp
- :func:`second()` -- retrieve the "second" component of a timestamp
- :func:`nanosecond()` -- retrieve the "nanosecond" component of a timestamp
)";


const char* doc_time_minute =
R"(minute(time)
--

.. x-version-added:: 1.0.0

Retrieve the "minute" component of a time64 column. The produced column
will have values in the range [0; 59].


Parameters
----------
time: FExpr[time64]
    A column for which you want to compute the minute part.

return: FExpr[int32]
    The minute part of the source column.


Examples
--------
>>> from datetime import datetime as d
>>> DT = dt.Frame([d(2020, 5, 11, 12, 0, 0), d(2021, 6, 14, 16, 10, 59, 394873)])
>>> DT[:, {'time': f[0], 'minute': dt.time.minute(f[0])}]
   | time                        minute
   | time64                       int32
-- + --------------------------  ------
 0 | 2020-05-11T12:00:00              0
 1 | 2021-06-14T16:10:59.394873      10
[2 rows x 2 columns]


See Also
--------
- :func:`hour()` -- retrieve the "hour" component of a timestamp
- :func:`second()` -- retrieve the "second" component of a timestamp
- :func:`nanosecond()` -- retrieve the "nanosecond" component of a timestamp
)";


const char* doc_time_month =
R"(month(date)
--

.. x-version-added:: 1.0.0

Retrieve the "month" component of a date32 or time64 column.


Parameters
----------
date: FExpr[date32] | FExpr[time64]
    A column for which you want to compute the month part.

return: FExpr[int32]
    The month part of the source column.


Examples
--------
>>> DT = dt.Frame([1, 1000, 100000], stype='date32')
>>> DT[:, {'date': f[0], 'month': dt.time.month(f[0])}]
   | date        month
   | date32      int32
-- + ----------  -----
 0 | 1970-01-02      1
 1 | 1972-09-27      9
 2 | 2243-10-17     10
[3 rows x 2 columns]


See Also
--------
- :func:`year()` -- retrieve the "year" component of a date
- :func:`day()` -- retrieve the "day" component of a date
)";


const char* doc_time_nanosecond =
R"(nanosecond(time)
--

.. x-version-added:: 1.0.0

Retrieve the "nanosecond" component of a time64 column. The produced
column will have values in the range [0; 999999999].


Parameters
----------
time: FExpr[time64]
    A column for which you want to compute the nanosecond part.

return: FExpr[int32]
    The "nanosecond" part of the source column.


Examples
--------
>>> from datetime import datetime as d
>>> DT = dt.Frame([d(2020, 5, 11, 12, 0, 0), d(2021, 6, 14, 16, 10, 59, 394873)])
>>> DT[:, {'time': f[0], 'ns': dt.time.nanosecond(f[0])}]
   | time                               ns
   | time64                          int32
-- + --------------------------  ---------
 0 | 2020-05-11T12:00:00                 0
 1 | 2021-06-14T16:10:59.394873  394873000
[2 rows x 2 columns]


See Also
--------
- :func:`hour()` -- retrieve the "hour" component of a timestamp
- :func:`minute()` -- retrieve the "minute" component of a timestamp
- :func:`second()` -- retrieve the "second" component of a timestamp
)";


const char* doc_time_second =
R"(second(time)
--

.. x-version-added:: 1.0.0

Retrieve the "second" component of a time64 column. The produced
column will have values in the range [0; 59].


Parameters
----------
time: FExpr[time64]
    A column for which you want to compute the second part.

return: FExpr[int32]
    The "second" part of the source column.


Examples
--------
>>> from datetime import datetime as d
>>> DT = dt.Frame([d(2020, 5, 11, 12, 0, 0), d(2021, 6, 14, 16, 10, 59, 394873)])
>>> DT[:, {'time': f[0], 'second': dt.time.second(f[0])}]
   | time                        second
   | time64                       int32
-- + --------------------------  ------
 0 | 2020-05-11T12:00:00              0
 1 | 2021-06-14T16:10:59.394873      59
[2 rows x 2 columns]


See Also
--------
- :func:`hour()` -- retrieve the "hour" component of a timestamp
- :func:`minute()` -- retrieve the "minute" component of a timestamp
- :func:`nanosecond()` -- retrieve the "nanosecond" component of a timestamp
)";


const char* doc_time_year =
R"(year(date)
--

.. x-version-added:: 1.0.0

Retrieve the "year" component of a date32 or time64 column.


Parameters
----------
date: FExpr[date32] | FExpr[time64]
    A column for which you want to compute the year part.

return: FExpr[int32]
    The year part of the source column.


Examples
--------
>>> DT = dt.Frame([1, 1000, 100000], stype='date32')
>>> DT[:, {'date': f[0], 'year': dt.time.year(f[0])}]
   | date         year
   | date32      int32
-- + ----------  -----
 0 | 1970-01-02   1970
 1 | 1972-09-27   1972
 2 | 2243-10-17   2243
[3 rows x 2 columns]


See Also
--------
- :func:`month()` -- retrieve the "month" component of a date
- :func:`day()` -- retrieve the "day" component of a date
)";


const char* doc_time_ymd =
R"(ymd(year, month, day)
--

.. x-version-added:: 1.0.0

Create a date32 column out of `year`, `month` and `day` components.

This function performs range checks on `month` and `day` columns: if a
certain combination of year/month/day is not valid in the Gregorian
calendar, then an NA value will be produced in that row.


Parameters
----------
year: FExpr[int]
    The year part of the resulting date32 column.

month: FExpr[int]
    The month part of the resulting date32 column. Values in this column
    are expected to be in the 1 .. 12 range.

day: FExpr[int]
    The day part of the resulting date32 column. Values in this column
    should be from 1 to ``last_day_of_month(year, month)``.

return: FExpr[date32]


Examples
--------
>>> DT = dt.Frame(y=[2005, 2010, 2015], m=[2, 3, 7])
>>> DT[:, dt.time.ymd(f.y, f.m, 30)]
   | C0
   | date32
-- + ----------
 0 | NA
 1 | 2010-03-30
 2 | 2015-07-30
[3 rows x 1 column]
)";


const char* doc_time_ymdt =
R"(ymdt(year, month, day, hour, minute, second, nanosecond=0, *, date=None)
--

.. x-version-added:: 1.0.0

Create a time64 column out of `year`, `month`, `day`, `hour`, `minute`,
`second` and `nanosecond` (optional) components. Alternatively, instead
of year-month-day triple you can pass `date` argument of type `date32`.

This function performs range checks on `month` and `day` columns: if a
certain combination of year/month/day is not valid in the Gregorian
calendar, then an NA value will be produced in that row.

At the same time, there are no range checks for time components. Thus,
you can, for example, pass `second=3600` instead of `hour=1`.


Parameters
----------
year: FExpr[int]
    The year part of the resulting time64 column.

month: FExpr[int]
    The month part of the resulting time64 column. Values in this column
    must be in the 1 .. 12 range.

day: FExpr[int]
    The day part of the resulting time64 column. Values in this column
    should be from 1 to ``last_day_of_month(year, month)``.

hour: FExpr[int]
    The hour part of the resulting time64 column.

minute: FExpr[int]
    The minute part of the resulting time64 column.

second: FExpr[int]
    The second part of the resulting time64 column.

nanosecond: FExpr[int]
    The nanosecond part of the resulting time64 column. This parameter
    is optional.

date: FExpr[date32]
    The date component of the resulting time64 column. This parameter,
    if given, replaces parameters `year`, `month` and `day`, and cannot
    be used together with them.

return: FExpr[time64]


Examples
--------
>>> DT = dt.Frame(Y=[2001, 2003, 2005, 2020, 1960],
...               M=[1, 5, 4, 11, 8],
...               D=[12, 18, 30, 1, 14],
...               h=[7, 14, 22, 23, 12],
...               m=[15, 30, 0, 59, 0],
...               s=[12, 23, 0, 59, 27],
...               ns=[0, 0, 0, 999999000, 123000])
>>> DT[:, [f[:], dt.time.ymdt(f.Y, f.M, f.D, f.h, f.m, f.s, f.ns)]]
   |     Y      M      D      h      m      s         ns  C0
   | int32  int32  int32  int32  int32  int32      int32  time64
-- + -----  -----  -----  -----  -----  -----  ---------  --------------------------
 0 |  2001      1     12      7     15     12          0  2001-01-12T07:15:12
 1 |  2003      5     18     14     30     23          0  2003-05-18T14:30:23
 2 |  2005      4     30     22      0      0          0  2005-04-30T22:00:00
 3 |  2020     11      1     23     59     59  999999000  2020-11-01T23:59:59.999999
 4 |  1960      8     14     12      0     27     123000  1960-08-14T12:00:27.000123
[5 rows x 8 columns]
)";


const char* doc_Frame =
R"(Two-dimensional column-oriented container of data. This the primary
data structure in the :mod:`datatable` module.

A Frame is *two-dimensional* in the sense that it is comprised of
rows and columns of data. Each data cell can be located via a pair
of its coordinates: ``(irow, icol)``. We do not support frames with
more or less than two dimensions.

A Frame is *column-oriented* in the sense that internally the data is
stored separately for each column. Each column has its own name and
type. Types may be different for different columns but cannot vary
within each column.

Thus, the dimensions of a Frame are not symmetrical: a Frame is not
a matrix. Internally the class is optimized for the use case when the
number of rows significantly exceeds the number of columns.

A Frame can be viewed as a ``list`` of columns: standard Python
function ``len()`` will return the number of columns in the Frame,
and ``frame[j]`` will return the column at index ``j`` (each "column"
will be a Frame with ``ncols == 1``). Similarly, you can iterate over
the columns of a Frame in a loop, or use it in a ``*``-expansion::

    >>> for column in frame:
    ...    # do something
    ...
    >>> list_of_columns = [*frame]

A Frame can also be viewed as a ``dict`` of columns, where the key
associated with each column is its name. Thus, ``frame[name]`` will
return the column with the requested name. A Frame can also work with
standard python ``**``-expansion::

    >>> dict_of_columns = {**frame}


Construction
------------

.. list-table::
    :widths: auto
    :class: api-table

    * - :meth:`Frame(*args, **kws) <dt.Frame.__init__>`
      - Construct the frame from various Python sources.

    * - :func:`dt.fread(src)`
      - Read an external file and convert into a Frame.

    * - :meth:`.copy()`
      - Create a copy of the frame.


Properties
----------

.. list-table::
    :widths: auto
    :class: api-table

    * - :attr:`.key`
      - The primary key for the Frame, if any.

    * - :attr:`.ltypes`
      - Logical types (:class:`dt.ltype`s) of all columns.

    * - :attr:`.meta`
      - The frame's meta information.

    * - :attr:`.names`
      - The names of all columns in the frame.

    * - :attr:`.ncols`
      - Number of columns in the frame.

    * - :attr:`.nrows`
      - Number of rows in the frame.

    * - :attr:`.stype`
      - A tuple (number of rows, number of columns).

    * - :attr:`.source`
      - Where this frame was loaded from.

    * - :attr:`.stype`
      - The common :class:`dt.stype` for the entire frame.

    * - :attr:`.stypes`
      - Storage types (:class:`dt.stype`s) of all columns.

    * - :attr:`.type`
      - The common type (:class:`dt.Type`) for the entire frame.

    * - :attr:`.types`
      - types (:class:`dt.Type`s) of all columns.


Frame manipulation
------------------

.. list-table::
    :widths: auto
    :class: api-table

    * - :meth:`frame[i, j, ...] <datatable.Frame.__getitem__>`
      - Primary method for extracting data from a frame.

    * - :meth:`frame[i, j, ...] = values <datatable.Frame.__setitem__>`
      - Update data within the frame.

    * - :meth:`del frame[i, j, ...] <datatable.Frame.__delitem__>`
      - Remove rows/columns/values from the frame.

    * - :meth:`.cbind(*frames)`
      - Append columns of other frames to this frame.

    * - :meth:`.rbind(*frames)`
      - Append other frames at the bottom of the current.

    * - :meth:`.replace(what, with)`
      - Search and replace values in the frame.

    * - :meth:`.sort(cols)`
      - Sort the frame by the specified columns.


.. _convert-into-other-formats:

Convert into other formats
--------------------------

.. list-table::
    :widths: auto
    :class: api-table

    * - :meth:`.to_arrow()`
      - Convert the frame into an Arrow table.

    * - :meth:`.to_csv(file)`
      - Write the frame's data into CSV format.

    * - :meth:`.to_dict()`
      - Convert the frame into a Python dictionary, by columns.

    * - :meth:`.to_jay(file)`
      - Store the frame's data into a binary file in Jay format.

    * - :meth:`.to_list()`
      - Return the frame's data as a list of lists, by columns.

    * - :meth:`.to_numpy()`
      - Convert the frame into a numpy array.

    * - :meth:`.to_pandas()`
      - Convert the frame into a pandas DataFrame.

    * - :meth:`.to_tuples()`
      - Return the frame's data as a list of tuples, by rows.


Statistical methods
-------------------

.. list-table::
    :widths: auto
    :class: api-table

    * - :meth:`.countna()`
      - Count missing values for each column in the frame.

    * - :meth:`.countna1()`
      - Count missing values for a one-column frame and return it as a scalar.

    * - :meth:`.kurt()`
      - Calculate excess kurtosis for each column in the frame.

    * - :meth:`.kurt1()`
      - Calculate excess kurtosis for a one-column frame and return it as a scalar.

    * - :meth:`.max()`
      - Find the largest element for each column in the frame.

    * - :meth:`.max1()`
      - Find the largest element for a one-column frame and return it as a scalar.

    * - :meth:`.mean()`
      - Calculate the mean value for each column in the frame.

    * - :meth:`.mean1()`
      - Calculate the mean value for a one-column frame and return it as a scalar.

    * - :meth:`.min()`
      - Find the smallest element for each column in the frame.

    * - :meth:`.min1()`
      - Find the smallest element for a one-column frame and return it as a scalar.

    * - :meth:`.mode()`
      - Find the mode value for each column in the frame.

    * - :meth:`.mode1()`
      - Find the mode value for a one-column frame and return it as a scalar.

    * - :meth:`.nmodal()`
      - Calculate the modal frequency for each column in the frame.

    * - :meth:`.nmodal1()`
      - Calculate the modal frequency for a one-column frame and return it as a scalar.

    * - :meth:`.nunique()`
      - Count the number of unique values for each column in the frame.

    * - :meth:`.nunique1()`
      - Count the number of unique values for a one-column frame and return it as a scalar.

    * - :meth:`.sd()`
      - Calculate the standard deviation for each column in the frame.

    * - :meth:`.sd1()`
      - Calculate the standard deviation for a one-column frame and return it as a scalar.

    * - :meth:`.skew()`
      - Calculate skewness for each column in the frame.

    * - :meth:`.skew1()`
      - Calculate skewness for a one-column frame and return it as a scalar.

    * - :meth:`.sum()`
      - Calculate the sum of all values for each column in the frame.

    * - :meth:`.sum1()`
      - Calculate the sum of all values for a one-column column frame and return it as a scalar.


Miscellaneous methods
---------------------

.. list-table::
    :widths: auto
    :class: api-table

    * - :meth:`.colindex(name)`
      - Find the position of a column in the frame by its name.

    * - :meth:`.export_names()`
      - Create python variables for each column of the frame.

    * - :meth:`.head()`
      - Return the first few rows of the frame.

    * - :meth:`.materialize()`
      - Make sure all frame's data is physically written to memory.

    * - :meth:`.tail()`
      - Return the last few rows of the frame.


Special methods
---------------

These methods are not intended to be called manually, instead they provide
a way for :mod:`datatable` to interoperate with other Python modules or
builtin functions.

.. list-table::
    :widths: auto
    :class: api-table

    * - :meth:`.__copy__()`
      - Used by Python module :ext-mod:`copy`.

    * - :meth:`.__deepcopy__() <dt.Frame.__copy__>`
      - Used by Python module :ext-mod:`copy`.

    * - :meth:`.__delitem__()`
      - Method that implements the ``del DT[...]`` call.

    * - :meth:`.__getitem__()`
      - Method that implements the ``DT[...]`` call.

    * - :meth:`.__getstate__()`
      - Used by Python module :ext-mod:`pickle`.

    * - :meth:`.__init__(...)`
      - The constructor function.

    * - :meth:`.__iter__()`
      - Used by Python function :ext-func:`iter() <iter>`, or when the frame
        is used as a target in a loop.

    * - :meth:`.__len__()`
      - Used by Python function :ext-func:`len() <len>`.

    * - :meth:`.__repr__()`
      - Used by Python function :ext-func:`repr() <repr>`.

    * - :meth:`.__reversed__()`
      - Used by Python function :ext-func:`reversed() <reversed>`.

    * - :meth:`.__setitem__()`
      - Method that implements the ``DT[...] = expr`` call.

    * - :meth:`.__setstate__() <dt.Frame.__getstate__>`
      - Used by Python module :ext-mod:`pickle`.

    * - :meth:`.__sizeof__()`
      - Used by :ext-func:`sys.getsizeof`.

    * - :meth:`.__str__()`
      - Used by Python function :ext-class:`str`.

    * - ``._repr_html_()``
      - Used to display the frame in `Jupyter Lab`_.

    * - ``._repr_pretty_()``
      - Used to display the frame in an :ext-mod:`IPython` console.
)";


const char* doc_Frame___init__ =
R"(__init__(self, _data=None, *, names=None, types=None,
--

Create a new Frame from a single or multiple sources.

Argument `_data` (or `**cols`) contains the source data for Frame's
columns. Column names are either derived from the data, given
explicitly via the `names` argument, or generated automatically.
Either way, the constructor ensures that column names are unique,
non-empty, and do not contain certain special characters (see
:ref:`name-mangling` for details).

Parameters
----------
_data: Any
    The first argument to the constructor represents the source from
    which to construct the Frame. If this argument is given, then the
    varkwd arguments `**cols` should not be used.

    This argument can accept a wide range of data types; see the
    "Details" section below.

**cols: Any
    Sequence of varkwd column initializers. The keys become column
    names, and the values contain column data. Using varkwd arguments
    is equivalent to passing a ``dict`` as the `_data` argument.

    When varkwd initializers are used, the `names` parameter may not
    be given.

names: List[str|None]
    Explicit list (or tuple) of column names. The number of elements
    in the list must be the same as the number of columns being
    constructed.

    This parameter should not be used when constructing the frame
    from `**cols`.

types: List[Type] | Dict[str, Type]
    Explicit list (or dict) of column types. The number of elements
    in the list must be the same as the number of columns being
    constructed.

type: Type | type
    Similar to `types`, but provide a single type that will be used
    for all columns. This option cannot be used together with `types`.

return: Frame
    A :class:`Frame <datatable.Frame>` object is constructed and
    returned.

except: ValueError
    The exception is raised if the lengths of `names` or `types`
    lists are different from the number of columns created, or when
    creating several columns and they have incompatible lengths.


Details
-------
The shape of the constructed Frame depends on the type of the source
argument `_data` (or `**cols`). The argument `_data` and varkwd
arguments `**cols` are mutually exclusive: they cannot be used at the
same time. However, it is possible to use neither and construct an
empty frame::

    >>> dt.Frame()       # empty 0x0 frame
    >>> dt.Frame(None)   # same
    >>> dt.Frame([])     # same

The varkwd arguments `**cols` can be used to construct a Frame by
columns. In this case the keys become column names, and the values
are column initializers. This form is mostly used for convenience;
it is equivalent to converting `cols` into a `dict` and passing as
the first argument::

    >>> dt.Frame(A = range(7),
    ...          B = [0.1, 0.3, 0.5, 0.7, None, 1.0, 1.5],
    ...          C = ["red", "orange", "yellow", "green", "blue", "indigo", "violet"])
    >>> # equivalent to
    >>> dt.Frame({"A": range(7),
    ...           "B": [0.1, 0.3, 0.5, 0.7, None, 1.0, 1.5],
    ...           "C": ["red", "orange", "yellow", "green", "blue", "indigo", "violet"]})
       |     A        B  C
       | int32  float64  str32
    -- + -----  -------  ------
     0 |     0      0.1  red
     1 |     1      0.3  orange
     2 |     2      0.5  yellow
     3 |     3      0.7  green
     4 |     4     NA    blue
     5 |     5      1    indigo
     6 |     6      1.5  violet
    [7 rows x 3 columns]

The argument `_data` accepts a wide range of input types. The
following list describes possible choices:

``List[List | Frame | np.array | pd.DataFrame | pd.Series | range | typed_list]``
    When the source is a non-empty list containing other lists or
    compound objects, then each item will be interpreted as a column
    initializer, and the resulting frame will have as many columns
    as the number of items in the list.

    Each element in the list must produce a single column. Thus,
    it is not allowed to use multi-column `Frame`s, or
    multi-dimensional numpy arrays or pandas `DataFrame`s.

        >>> dt.Frame([[1, 3, 5, 7, 11],
        ...           [12.5, None, -1.1, 3.4, 9.17]])
           |    C0       C1
           | int32  float64
        -- + -----  -------
         0 |     1    12.5
         1 |     3    NA
         2 |     5    -1.1
         3 |     7     3.4
         4 |    11     9.17
        [5 rows x 2 columns]

    Note that unlike `pandas` and `numpy`, we treat a list of lists
    as a list of columns, not a list of rows. If you need to create
    a Frame from a row-oriented store of data, you can use a list of
    dictionaries or a list of tuples as described below.

``List[Dict]``
    If the source is a list of `dict` objects, then each element
    in this list is interpreted as a single row. The keys
    in each dictionary are column names, and the values contain
    contents of each individual cell.

    The rows don't have to have the same number or order of
    entries: all missing elements will be filled with NAs::

        >>> dt.Frame([{"A": 3, "B": 7},
        ...           {"A": 0, "B": 11, "C": -1},
        ...           {"C": 5}])
           |     A      B      C
           | int32  int32  int32
        -- + -----  -----  -----
         0 |     3      7     NA
         1 |     0     11     -1
         2 |    NA     NA      5
        [3 rows x 3 columns]

    If the `names` parameter is given, then only the keys given
    in the list of names will be taken into account, all extra
    fields will be discarded.

``List[Tuple]``
    If the source is a list of `tuple`s, then each tuple
    represents a single row. The tuples must have the same size,
    otherwise an exception will be raised::

        >>> dt.Frame([(39, "Mary"),
        ...           (17, "Jasmine"),
        ...           (23, "Lily")], names=['age', 'name'])
           |   age  name
           | int32  str32
        -- + -----  -------
         0 |    39  Mary
         1 |    17  Jasmine
         2 |    23  Lily
        [3 rows x 2 columns]

    If the tuples are in fact `namedtuple`s, then the field names
    will be used for the column names in the resulting Frame. No
    check is made whether the named tuples in fact belong to the
    same class.

``List[Any]``
    If the list's first element does not match any of the cases
    above, then it is considered a "list of primitives". Such list
    will be parsed as a single column.

    The entries are typically `bool`s, `int`s, `float`s, `str`s,
    or `None`s; numpy scalars are also allowed. If the list has
    elements of heterogeneous types, then we will attempt to
    convert them to the smallest common stype.

    If the list contains only boolean values (or `None`s), then it
    will create a column of type `bool8`.

    If the list contains only integers (or `None`s), then the
    resulting column will be `int8` if all integers are 0 or 1; or
    `int32` if all entries are less than :math:`2^{31}` in magnitude;
    otherwise `int64` if all entries are less than :math:`2^{63}`
    in magnitude; or otherwise `float64`.

    If the list contains floats, then the resulting column will have
    stype `float64`. Both `None` and `math.nan` can be used to input
    NA values.

    Finally, if the list contains strings then the column produced
    will have stype `str32` if the total size of the character is
    less than 2Gb, or `str64` otherwise.

``typed_list``
    A typed list can be created by taking a regular list and
    dividing it by an stype. It behaves similarly to a simple
    list of primitives, except that it is parsed into the specific
    stype.

        >>> dt.Frame([1.5, 2.0, 3.87] / dt.float32).type
        Type.float32

``Dict[str, Any]``
    The keys are column names, and values can be any objects from
    which a single-column frame can be constructed: list, range,
    np.array, single-column Frame, pandas series, etc.

    Constructing a frame from a dictionary `d` is exactly equivalent
    to calling `dt.Frame(list(d.values()), names=list(d.keys()))`.

``range``
    Same as if the range was expanded into a list of integers,
    except that the column created from a range is virtual and
    its creation time is nearly instant regardless of the range's
    length.

``Frame``
    If the argument is a :class:`Frame <datatable.Frame>`, then
    a shallow copy of that frame will be created, same as
    :meth:`.copy()`.

``str``
    If the source is a simple string, then the frame is created
    by :func:`fread <datatable.fread>`-ing this string.
    In particular, if the string contains the name of a file, the
    data will be loaded from that file; if it is a URL, the data
    will be downloaded and parsed from that URL. Lastly, the
    string may simply contain a table of data.

        >>> DT1 = dt.Frame("train.csv")
        >>> dt.Frame("""
        ...    Name    Age
        ...    Mary     39
        ...    Jasmine  17
        ...    Lily     23 """)
           | Name       Age
           | str32    int32
        -- + -------  -----
         0 | Mary        39
         1 | Jasmine     17
         2 | Lily        NA
        [3 rows x 2 columns]

``pd.DataFrame | pd.Series``
    A pandas DataFrame (Series) will be converted into a datatable
    Frame. Column names will be preserved.

    Column types will generally be the same, assuming they have a
    corresponding stype in datatable. If not, the column will be
    converted. For example, pandas date/time column will get converted
    into string, while `float16` will be converted into `float32`.

    If a pandas frame has an object column, we will attempt to refine
    it into a more specific stype. In particular, we can detect a
    string or boolean column stored as object in pandas.

``np.array``
    A numpy array will get converted into a Frame of the same shape
    (provided that it is 2- or less- dimensional) and the same type.

    If possible, we will create a Frame without copying the data
    (however, this is subject to numpy's approval). The resulting
    frame will have a copy-on-write semantics.

``pyarrow.Table``
    An arrow table will be converted into a datatable Frame, preserving
    column names and types.

    If the arrow table has columns of types not supported by datatable
    (for example lists or structs), an exception will be raised.

``None``
    When the source is not given at all, then a 0x0 frame will be
    created; unless a `names` parameter is provided, in which
    case the resulting frame will have 0 rows but as many columns
    as given in the `names` list.
)";


const char* doc_Frame___sizeof__ =
R"(__sizeof__(self)
--

Return the size of this Frame in memory.

The function attempts to compute the total memory size of the frame
as precisely as possible. In particular, it takes into account not
only the size of data in columns, but also sizes of all auxiliary
internal structures.

Special cases: if frame is a view (say, `d2 = DT[:1000, :]`), then
the reported size will not contain the size of the data, because that
data "belongs" to the original datatable and is not copied. However if
a frame selects only a subset of columns (say, `d3 = DT[:, :5]`),
then a view is not created and instead the columns are copied by
reference. Frame `d3` will report the "full" size of its columns,
even though they do not occupy any extra memory compared to `DT`.
This behavior may be changed in the future.

This function is not intended for manual use. Instead, in order to
get the size of a frame `DT`, call `sys.getsizeof(DT)`.
)";


const char* doc_Frame_cbind =
R"(cbind(self, *frames, force=False)
--

Append columns of one or more `frames` to the current Frame.

For example, if the current frame has `n` columns, and you are
appending another frame with `k` columns, then after this method
succeeds, the current frame will have `n + k` columns. Thus, this
method is roughly equivalent to `pandas.concat(axis=1)`.

The frames being cbound must all either have the same number of rows,
or some of them may have only a single row. Such single-row frames
will be automatically expanded, replicating the value as needed.
This makes it easy to create constant columns or to append reduction
results (such as min/max/mean/etc) to the current Frame.

If some of the `frames` have an incompatible number of rows, then the
operation will fail with an :exc:`dt.exceptions.InvalidOperationError`.
However, if you set the flag `force` to True, then the error will no
longer be raised - instead all frames that are shorter than the others
will be padded with NAs.

If the frames being appended have the same column names as the current
frame, then those names will be :ref:`mangled <name-mangling>`
to ensure that the column names in the current frame remain unique.
A warning will also be issued in this case.


Parameters
----------
frames: Frame | List[Frame] | None
    The list/tuple/sequence/generator expression of Frames to append
    to the current frame. It may also contain `None` values,
    which will be simply skipped.

force: bool
    If `True`, allows Frames to be appended even if they have unequal
    number of rows. The resulting Frame will have number of rows equal
    to the largest among all Frames. Those Frames which have less
    than the largest number of rows, will be padded with NAs (with the
    exception of Frames having just 1 row, which will be replicated
    instead of filling with NAs).

return: None
    This method alters the current frame in-place, and doesn't return
    anything.

except: InvalidOperationError
    If trying to cbind frames with the number of rows different from
    the current frame's, and the option `force` is not set.


Notes
-----

Cbinding frames is a very cheap operation: the columns are copied by
reference, which means the complexity of the operation depends only
on the number of columns, not on the number of rows. Still, if you
are planning to cbind a large number of frames, it will be beneficial
to collect them in a list first and then call a single `cbind()`
instead of cbinding them one-by-one.

It is possible to cbind frames using the standard `DT[i,j]` syntax::

    >>> df[:, update(**frame1, **frame2, ...)]

Or, if you need to append just a single column::

    >>> df["newcol"] = frame1


See also
--------
- :func:`datatable.cbind` -- function for cbinding frames
  "out-of-place" instead of in-place;

- :meth:`.rbind()` -- method for row-binding frames.


Examples
--------
>>> DT = dt.Frame(A=[1, 2, 3], B=[4, 7, 0])
>>> frame1 = dt.Frame(N=[-1, -2, -5])
>>> DT.cbind(frame1)
>>> DT
   |     A      B      N
   | int32  int32  int32
-- + -----  -----  -----
 0 |     1      4     -1
 1 |     2      7     -2
 2 |     3      0     -5
[3 rows x 3 columns]
)";


const char* doc_Frame_colindex =
R"(colindex(self, column)
--

Return the position of the `column` in the Frame.

The index of the first column is `0`, just as with regular python
lists.


Parameters
----------
column: str | int | FExpr
    If string, then this is the name of the column whose index you
    want to find.

    If integer, then this represents a column's index. The return
    value is thus the same as the input argument `column`, provided
    that it is in the correct range. If the `column` argument is
    negative, then it is interpreted as counting from the end of the
    frame. In this case the positive value `column + ncols` is
    returned.

    Lastly, `column` argument may also be an
    :ref:`f-expression <f-expressions>` such as `f.A` or `f[3]`. This
    case is treated as if the argument was simply `"A"` or `3`. More
    complicated f-expressions are not allowed and will result in a
    `TypeError`.

return: int
    The numeric index of the provided `column`. This will be an
    integer between `0` and `self.ncols - 1`.

except: KeyError | IndexError
    .. list-table::
        :widths: auto
        :class: api-table

        * - :exc:`dt.exceptions.KeyError`
          - raised if the `column` argument is a string, and the column with
            such name does not exist in the frame. When this exception is
            thrown, the error message may contain suggestions for up to 3
            similarly looking column names that actually exist in the Frame.

        * - :exc:`dt.exceptions.IndexError`
          - raised if the `column` argument is an integer that is either greater
            than or equal to :attr:`.ncols` or less than `-ncols`.

Examples
--------

>>> df = dt.Frame(A=[3, 14, 15], B=["enas", "duo", "treis"],
...               C=[0, 0, 0])
>>> df.colindex("B")
1
>>> df.colindex(-1)
2

>>> from datatable import f
>>> df.colindex(f.A)
0
)";


const char* doc_Frame_copy =
R"(copy(self, deep=False)
--

Make a copy of the frame.

The returned frame will be an identical copy of the original,
including column names, types, and keys.

By default, copying is shallow with copy-on-write semantics. This
means that only the minimal information about the frame is copied,
while all the internal data buffers are shared between the copies.
Nevertheless, due to the copy-on-write semantics, any changes made to
one of the frames will not propagate to the other; instead, the data
will be copied whenever the user attempts to modify it.

It is also possible to explicitly request a deep copy of the frame
by setting the parameter `deep` to `True`. With this flag, the
returned copy will be truly independent from the original. The
returned frame will also be fully materialized in this case.


Parameters
----------
deep: bool
    Flag indicating whether to return a "shallow" (default), or a
    "deep" copy of the original frame.

return: Frame
    A new Frame, which is the copy of the current frame.


Examples
--------

>>> DT1 = dt.Frame(range(5))
>>> DT2 = DT1.copy()
>>> DT2[0, 0] = -1
>>> DT2
   |    C0
   | int32
-- + -----
 0 |    -1
 1 |     1
 2 |     2
 3 |     3
 4 |     4
[5 rows x 1 column]
>>> DT1
   |    C0
   | int32
-- + -----
 0 |     0
 1 |     1
 2 |     2
 3 |     3
 4 |     4
[5 rows x 1 column]


Notes
-----
- Non-deep frame copy is a very low-cost operation: its speed depends
  on the number of columns only, not on the number of rows. On a
  regular laptop copying a 100-column frame takes about 30-50µs.

- Deep copying is more expensive, since the data has to be physically
  written to new memory, and if the source columns are virtual, then
  they need to be computed too.

- Another way to create a copy of the frame is using a `DT[i, j]`
  expression (however, this will not copy the key property)::

    >>> DT[:, :]

- `Frame` class also supports copying via the standard Python library
  ``copy``::

    >>> import copy
    >>> DT_shallow_copy = copy.copy(DT)
    >>> DT_deep_copy = copy.deepcopy(DT)
)";


const char* doc_Frame_countna1 =
R"(countna1(self)
--

Return the number of NA values in a single-column Frame.

This function is a shortcut for::

    DT.countna()[0, 0]

Parameters
----------
(except): ValueError
    If called on a Frame that has more or less than one column.

(return): int

See Also
--------
- :meth:`.countna()` -- similar to this method, but can be applied to
  a Frame with an arbitrary number of columns.

- :func:`dt.count()` -- function for counting non-NA ("valid") values
  in a column; can also be applied per-group.
)";


const char* doc_Frame_countna =
R"(countna(self)
--

Report the number of NA values in each column of the frame.

Parameters
----------
(return): Frame
    The frame will have one row and the same number/names of columns
    as in the current frame. All columns will have stype ``int64``.

Examples
--------
.. code-block:: python

    >>> DT = dt.Frame(A=[1, 5, None], B=[math.nan]*3, C=[None, None, 'bah!'])
    >>> DT.countna()
       |     A      B      C
       | int64  int64  int64
    -- + -----  -----  -----
     0 |     1      3      2
    [1 row x 3 columns]

    >>> DT.countna().to_tuples()[0]
    >>> (1, 3, 2)


See Also
--------
- :meth:`.countna1()` -- similar to this method, but operates on a
  single-column frame only, and returns a number instead of a Frame.

- :func:`dt.count()` -- function for counting non-NA ("valid") values
  in a column; can also be applied per-group.
)";


const char* doc_Frame_export_names =
R"(export_names(self)
--

.. x-version-added:: 0.10

Return a tuple of :ref:`f-expressions` for all columns of the frame.

For example, if the frame has columns "A", "B", and "C", then this
method will return a tuple of expressions ``(f.A, f.B, f.C)``. If you
assign these to, say, variables ``A``, ``B``, and ``C``, then you
will be able to write column expressions using the column names
directly, without using the ``f`` symbol::

    >>> A, B, C = DT.export_names()
    >>> DT[A + B > C, :]

The variables that are "exported" refer to each column *by name*. This
means that you can use the variables even after reordering the
columns. In addition, the variables will work not only for the frame
they were exported from, but also for any other frame that has columns
with the same names.

Parameters
----------
return: Tuple[Expr, ...]
    The length of the tuple is equal to the number of columns in the
    frame. Each element of the tuple is a datatable *expression*, and
    can be used primarily with the ``DT[i,j]`` notation.

Notes
-----
- This method is effectively equivalent to::

    >>> def export_names(self):
    ...     return tuple(f[name] for name in self.names)

- If you want to export only a subset of column names, then you can
  either subset the frame first, or use ``*``-notation to ignore the
  names that you do not plan to use::

    >>> A, B = DT[:, :2].export_names()  # export the first two columns
    >>> A, B, *_ = DT.export_names()     # same

- Variables that you use in code do not have to have the same names
  as the columns::

    >>> Price, Quantity = DT[:, ["sale price", "quant"]].export_names()
)";


const char* doc_Frame_head =
R"(head(self, n=10)
--

Return the first `n` rows of the frame.

If the number of rows in the frame is less than `n`, then all rows
are returned.

This is a convenience function and it is equivalent to `DT[:n, :]`.


Parameters
----------
n : int
    The maximum number of rows to return, 10 by default. This number
    cannot be negative.

return: Frame
    A frame containing the first up to `n` rows from the original
    frame, and same columns.


Examples
--------
>>> DT = dt.Frame(A=["apples", "bananas", "cherries", "dates",
...                  "eggplants", "figs", "grapes", "kiwi"])
>>> DT.head(4)
   | A
   | str32
-- + --------
 0 | apples
 1 | bananas
 2 | cherries
 3 | dates
[4 rows x 1 column]


See also
--------
- :meth:`.tail` -- return the last `n` rows of the Frame.
)";


const char* doc_Frame_key =
R"(The tuple of column names that are the primary key for this frame.

If the frame has no primary key, this property returns an empty tuple.

The primary key columns are always located at the beginning of the
frame, and therefore the following holds::

    >>> DT.key == DT.names[:len(DT.key)]

Assigning to this property will make the Frame keyed by the specified
column(s). The key columns will be moved to the front, and the Frame
will be sorted. The values in the key columns must be unique.

Parameters
----------
return: Tuple[str, ...]
    When used as a getter, returns the tuple of names of the primary
    key columns.

new_key: str | List[str] | Tuple[str, ...] | None
    Specify a column or a list of columns that will become the new
    primary key of the Frame. Object columns cannot be used for a key.
    The values in the key column must be unique; if multiple columns
    are assigned as the key, then their combined (tuple-like) values
    must be unique.

    If `new_key` is `None`, then this is equivalent to deleting the
    key. When the key is deleted, the key columns remain in the frame,
    they merely stop being marked as "key".

except: ValueError
    Raised when the values in the key column(s) are not unique.

except: KeyError
    Raised when `new_key` contains a column name that doesn't exist
    in the Frame.

Examples
--------
>>> DT = dt.Frame(A=range(5), B=['one', 'two', 'three', 'four', 'five'])
>>> DT.key = 'B'
>>> DT
B     |     A
str32 | int32
----- + -----
five  |     4
four  |     3
one   |     0
three |     2
two   |     1
[5 rows x 2 columns]
)";


const char* doc_Frame_kurt1 =
R"(kurt1(self)
--

Calculate the excess kurtosis for a one-column frame and
return it as a scalar.

This function is a shortcut for::

    DT.kurt()[0, 0]

Parameters
----------
return: None | float
    `None` is returned for non-numeric columns.

except: ValueError
    If called on a Frame that has more or less than one column.

See Also
--------
- :meth:`.kurt()` -- similar to this method, but can be applied to
  a Frame with an arbitrary number of columns.
)";


const char* doc_Frame_kurt =
R"(kurt(self)
--

Calculate the excess kurtosis for each column in the frame.

Parameters
----------
return: Frame
    The frame will have one row and the same number/names
    of columns as in the current frame. All the columns
    will have `float64` stype. For non-numeric columns
    this function returns NA values.

See Also
--------
- :meth:`.kurt1()` -- similar to this method, but operates on a
  single-column frame only, and returns a scalar value instead of
  a Frame.
)";


const char* doc_Frame_ltypes =
R"(.. x-version-deprecated:: 1.0.0

    This property is deprecated and will be removed in version 1.2.0.
    Please use :attr:`.types` instead.

The tuple of each column's ltypes ("logical types").

Parameters
----------
return: Tuple[ltype, ...]
    The length of the tuple is the same as the number of columns in
    the frame.

See also
--------
- :attr:`.stypes` -- tuple of columns' storage types
)";


const char* doc_Frame_materialize =
R"(materialize(self, to_memory=False)
--

Force all data in the Frame to be laid out physically.

In datatable, a Frame may contain "virtual" columns, i.e. columns
whose data is computed on-the-fly. This allows us to have better
performance for certain types of computations, while also reducing
the total memory footprint. The use of virtual columns is generally
transparent to the user, and datatable will materialize them as
needed.

However, there could be situations where you might want to materialize
your Frame explicitly. In particular, materialization will carry out
all delayed computations and break internal references on other
Frames' data. Thus, for example if you subset a large frame to create
a smaller subset, then the new frame will carry an internal reference
to the original, preventing it from being garbage-collected. However,
if you materialize the small frame, then the data will be physically
copied, allowing the original frame's memory to be freed.

Parameters
----------
to_memory: bool
    If True, then, in addition to de-virtualizing all columns, this
    method will also copy all memory-mapped columns into the RAM.

    When you open a Jay file, the Frame that is created will contain
    memory-mapped columns whose data still resides on disk. Calling
    ``.materialize(to_memory=True)`` will force the data to be loaded
    into the main memory. This may be beneficial if you are concerned
    about the disk speed, or if the file is on a removable drive, or
    if you want to delete the source file.

return: None
    This operation modifies the frame in-place.
)";


const char* doc_Frame_max1 =
R"(max1(self)
--

Return the largest value in a single-column Frame. The frame's
stype must be numeric.

This function is a shortcut for::

    DT.max()[0, 0]

Parameters
----------
return: bool | int | float
    The returned value corresponds to the stype of the frame.

except: ValueError
    If called on a Frame that has more or less than one column.

See Also
--------
- :meth:`.max()` -- similar to this method, but can be applied to
  a Frame with an arbitrary number of columns.

- :func:`dt.max()` -- function for counting max values in a column or
  an expression; can also be applied per-group.
)";


const char* doc_Frame_max =
R"(max(self)
--

Find the largest value in each column of the frame.

Parameters
----------
return: Frame
    The frame will have one row and the same number, names and stypes
    of columns as in the current frame. For string/object columns
    this function returns NA values.

See Also
--------
- :meth:`.max1()` -- similar to this method, but operates on a
  single-column frame only, and returns a scalar value instead of
  a Frame.

- :func:`dt.max()` -- function for finding largest values in a column or
  an expression; can also be applied per-group.
)";


const char* doc_Frame_mean1 =
R"(mean1(self)
--

Calculate the mean value for a single-column Frame.

This function is a shortcut for::

    DT.mean()[0, 0]

Parameters
----------
return: None | float
    `None` is returned for string/object columns.

except: ValueError
    If called on a Frame that has more or less than one column.

See Also
--------
- :meth:`.mean()` -- similar to this method, but can be applied to
  a Frame with an arbitrary number of columns.

- :func:`dt.mean()` -- function for calculatin mean values in a column or
  an expression; can also be applied per-group.
)";


const char* doc_Frame_mean =
R"(mean(self)
--

Calculate the mean value for each column in the frame.

Parameters
----------
return: Frame
    The frame will have one row and the same number/names
    of columns as in the current frame. All columns will have `float64`
    stype. For string/object columns this function returns NA values.

See Also
--------
- :meth:`.mean1()` -- similar to this method, but operates on a
  single-column frame only, and returns a scalar value instead of
  a Frame.

- :func:`dt.mean()` -- function for counting mean values in a column or
  an expression; can also be applied per-group.
)";


const char* doc_Frame_meta =
R"(.. x-version-added:: 1.0

Frame's meta information.

This property contains meta information, if any, as set by datatable
functions and methods. It is a settable property, so that users can also
update it with any information relevant to a particular frame.

It is not guaranteed that the existing meta information will be preserved
by the functions and methods called on the frame. In particular,
it is not preserved when exporting data into a Jay file or pickling the data.
This behavior may change in the future.

The default value for this property is `None`.


Parameters
----------
return: dict | None
    If the frame carries any meta information, the corresponding meta
    information dictionary is returned, `None` is returned otherwise.

new_meta: dict | None
    New meta information.
)";


const char* doc_Frame_min1 =
R"(min1(self)
--

Find the smallest value in a single-column Frame. The frame's
stype must be numeric.

This function is a shortcut for::

    DT.min()[0, 0]

Parameters
----------
return: bool | int | float
    The returned value corresponds to the stype of the frame.

except: ValueError
    If called on a Frame that has more or less than 1 column.

See Also
--------
- :meth:`.min()` -- similar to this method, but can be applied to
  a Frame with an arbitrary number of columns.

- :func:`dt.min()` -- function for counting min values in a column or
  an expression; can also be applied per-group.
)";


const char* doc_Frame_min =
R"(min(self)
--

Find the smallest value in each column of the frame.

Parameters
----------
return: Frame
    The frame will have one row and the same number, names and stypes
    of columns as in the current frame. For string/object columns
    this function returns NA values.

See Also
--------
- :meth:`.min1()` -- similar to this method, but operates on a
  single-column frame only, and returns a scalar value instead of
  a Frame.

- :func:`dt.min()` -- function for counting min values in a column or
  an expression; can also be applied per-group.
)";


const char* doc_Frame_mode1 =
R"(mode1(self)
--

Find the mode for a single-column Frame.

This function is a shortcut for::

    DT.mode()[0, 0]

Parameters
----------
return: bool | int | float | str | object
    The returned value corresponds to the stype of the column.

except: ValueError
    If called on a Frame that has more or less than one column.

See Also
--------
- :meth:`.mode()` -- similar to this method, but can be applied to
  a Frame with an arbitrary number of columns.
)";


const char* doc_Frame_mode =
R"(mode(self)
--

Find the mode for each column in the frame.

Parameters
----------
return: Frame
    The frame will have one row and the same number/names
    of columns as in the current frame.

See Also
--------
- :meth:`.mode1()` -- similar to this method, but operates on a
  single-column frame only, and returns a scalar value instead of
  a Frame.
)";


const char* doc_Frame_names =
R"(The tuple of names of all columns in the frame.

Each name is a non-empty string not containing any ASCII control
characters, and jointly the names are unique within the frame.

This property is also assignable: setting ``DT.names`` has the effect
of renaming the frame's columns without changing their order. When
renaming, the length of the new list of names must be the same as the
number of columns in the frame. It is also possible to rename just a
few of the columns by assigning a dictionary ``{oldname: newname}``.
Any column not listed in the dictionary will keep its old name.

When setting new column names, we will verify whether they satisfy
the requirements mentioned above. If not, a warning will be emitted
and the names will be automatically :ref:`mangled <name-mangling>`.

Parameters
----------
return: Tuple[str, ...]
    When used in getter form, this property returns the names of all
    frame's columns, as a tuple. The length of the tuple is equal to
    the number of columns in the frame, :attr:`.ncols`.

new_names: List[str?] | Tuple[str?, ...] | Dict[str, str?] | None
    The most common form is to assign the list or tuple of new
    column names. The length of the new list must be equal to the
    number of columns in the frame. Some (or all) elements in the list
    may be ``None``'s, indicating that that column should have
    an auto-generated name.

    If ``new_names`` is a dictionary, then it provides a mapping from
    old to new column names. The dictionary may contain less entries
    than the number of columns in the frame: the columns not mentioned
    in the dictionary will retain their names.

    Setting the ``.names`` to ``None`` is equivalent to using the
    ``del`` keyword: the names will be set to their default values,
    which are usually ``C0, C1, ...``.


except: ValueError | KeyError
    .. list-table::
        :widths: auto
        :class: api-table

        * - :exc:`dt.exceptions.ValueError`
          - raised If the length of the list/tuple `new_names` does not match the
            number of columns in the frame.

        * - :exc:`dt.exceptions.KeyError`
          - raised If `new_names` is a dictionary containing entries that do not
            match any of the existing columns in the frame.

Examples
--------
>>> DT = dt.Frame([[1], [2], [3]])
>>> DT.names = ['A', 'B', 'C']
>>> DT.names
('A', 'B', 'C')
>>> DT.names = {'B': 'middle'}
>>> DT.names
('A', 'middle', 'C')
>>> del DT.names
>>> DT.names
('C0', 'C1', 'C2)
)";


const char* doc_Frame_ncols =
R"(Number of columns in the frame.

Parameters
----------
return: int
    The number of columns can be either zero or a positive integer.

Notes
-----
The expression `len(DT)` also returns the number of columns in the
frame `DT`. Such usage, however, is not recommended.

See also
--------
- :attr:`.nrows`: getter for the number of rows of the frame.
)";


const char* doc_Frame_nmodal1 =
R"(nmodal1(self)
--

Calculate the modal frequency for a single-column Frame.

This function is a shortcut for::

    DT.nmodal()[0, 0]

Parameters
----------
return: int

except: ValueError
    If called on a Frame that has more or less than one column.

See Also
--------
- :meth:`.nmodal()` -- similar to this method, but can be applied to
  a Frame with an arbitrary number of columns.
)";


const char* doc_Frame_nmodal =
R"(nmodal(self)
--

Calculate the modal frequency for each column in the frame.

Parameters
----------
return: Frame
    The frame will have one row and the same number/names
    of columns as in the current frame. All the columns
    will have `int64` stype.

See Also
--------
- :meth:`.nmodal1()` -- similar to this method, but operates on a
  single-column frame only, and returns a scalar value instead of
  a Frame.
)";


const char* doc_Frame_nrows =
R"(Number of rows in the Frame.

Assigning to this property will change the height of the Frame,
either by truncating if the new number of rows is smaller than the
current, or filling with NAs if the new number of rows is greater.

Increasing the number of rows of a keyed Frame is not allowed.

Parameters
----------
return: int
    The number of rows can be either zero or a positive integer.

n: int
    The new number of rows for the frame, this should be a non-negative
    integer.

See also
--------
- :attr:`.ncols`: getter for the number of columns of the frame.
)";


const char* doc_Frame_nunique1 =
R"(nunique1(self)
--

Count the number of unique values for a one-column frame and return it as a scalar.

This function is a shortcut for::

    DT.nunique()[0, 0]

Parameters
----------
return: int

except: ValueError
    If called on a Frame that has more or less than one column.

See Also
--------
- :meth:`.nunique()` -- similar to this method, but can be applied to
  a Frame with an arbitrary number of columns.
)";


const char* doc_Frame_nunique =
R"(nunique(self)
--

Count the number of unique values for each column in the frame.

Parameters
----------
return: Frame
    The frame will have one row and the same number/names
    of columns as in the current frame. All the columns
    will have `int64` stype.

See Also
--------
- :meth:`.nunique1()` -- similar to this method, but operates on a
  single-column frame only, and returns a scalar value instead of
  a Frame.
)";


const char* doc_Frame_rbind =
R"(rbind(self, *frames, force=False, bynames=True)
--

Append rows of `frames` to the current frame.

This is equivalent to `list.extend()` in Python: the frames are
combined by rows, i.e. rbinding a frame of shape [n x k] to a Frame
of shape [m x k] produces a frame of shape [(m + n) x k].

This method modifies the current frame in-place. If you do not want
the current frame modified, then use the :func:`dt.rbind()` function.

If frame(s) being appended have columns of types different from the
current frame, then these columns will be promoted according to the
standard promotion rules. In particular, booleans can be promoted into
integers, which in turn get promoted into floats. However, they are
not promoted into strings or objects.

If frames have columns of incompatible types, a TypeError will be
raised.

If you need to append multiple frames, then it is more efficient to
collect them into an array first and then do a single `rbind()`, than
it is to append them one-by-one in a loop.

Appending data to a frame opened from disk will force loading the
current frame into memory, which may fail with an OutOfMemory
exception if the frame is sufficiently big.

Parameters
----------
frames: Frame | List[Frame]
    One or more frames to append. These frames should have the same
    columnar structure as the current frame (unless option `force` is
    used).

force: bool
    If True, then the frames are allowed to have mismatching set of
    columns. Any gaps in the data will be filled with NAs.

    In addition, when this parameter is True, rbind will no longer
    produce an error when combining columns of unrelated types.
    Instead, both columns will be converted into strings.

bynames: bool
    If True (default), the columns in frames are matched by their
    names. For example, if one frame has columns ["colA", "colB",
    "colC"] and the other ["colB", "colA", "colC"] then we will swap
    the order of the first two columns of the appended frame before
    performing the append. However if `bynames` is False, then the
    column names will be ignored, and the columns will be matched
    according to their order, i.e. i-th column in the current frame
    to the i-th column in each appended frame.

return: None
)";


const char* doc_Frame_replace =
R"(replace(self, replace_what, replace_with)
--

Replace given value(s) `replace_what` with `replace_with` in the entire Frame.

For each replace value, this method operates only on columns of types
appropriate for that value. For example, if `replace_what` is a list
`[-1, math.inf, None, "??"]`, then the value `-1` will be replaced in integer
columns only, `math.inf` only in real columns, `None` in columns of all types,
and finally `"??"` only in string columns.

The replacement value must match the type of the target being replaced,
otherwise an exception will be thrown. That is, a bool must be replaced with a
bool, an int with an int, a float with a float, and a string with a string.
The `None` value (representing NA) matches any column type, and therefore can
be used as either replacement target, or replace value for any column. In
particular, the following is valid: `DT.replace(None, [-1, -1.0, ""])`. This
will replace NA values in int columns with `-1`, in real columns with `-1.0`,
and in string columns with an empty string.

The replace operation never causes a column to change its logical type. Thus,
an integer column will remain integer, string column remain string, etc.
However, replacing may cause a column to change its stype, provided that
ltype remains constant. For example, replacing `0` with `-999` within an `int8`
column will cause that column to be converted into the `int32` stype.

Parameters
----------
replace_what: None, bool, int, float, list, or dict
    Value(s) to search for and replace.

replace_with: single value, or list
    The replacement value(s). If `replace_what` is a single value, then this
    must be a single value too. If `replace_what` is a list, then this could
    be either a single value, or a list of the same length. If `replace_what`
    is a dict, then this value should not be passed.

(return): None
    Nothing is returned, the replacement is performed in-place.


Examples
--------
>>> df = dt.Frame([1, 2, 3] * 3)
>>> df.replace(1, -1)
>>> df
   |    C0
   | int32
-- + -----
 0 |    -1
 1 |     2
 2 |     3
 3 |    -1
 4 |     2
 5 |     3
 6 |    -1
 7 |     2
 8 |     3
[9 rows x 1 column]

>>> df.replace({-1: 100, 2: 200, "foo": None})
>>> df
   |    C0
   | int32
-- + -----
 0 |   100
 1 |   200
 2 |     3
 3 |   100
 4 |   200
 5 |     3
 6 |   100
 7 |   200
 8 |     3
[9 rows x 1 column]
)";


const char* doc_Frame_sd1 =
R"(sd1(self)
--

Calculate the standard deviation for a one-column frame and
return it as a scalar.

This function is a shortcut for::

    DT.sd()[0, 0]

Parameters
----------
return: None | float
    `None` is returned for non-numeric columns.

except: ValueError
    If called on a Frame that has more or less than one column.

See Also
--------
- :meth:`.sd()` -- similar to this method, but can be applied to
  a Frame with an arbitrary number of columns.

- :func:`dt.sd()` -- function for calculating the standard deviation
  in a column or an expression; can also be applied per-group.
)";


const char* doc_Frame_sd =
R"(sd(self)
--

Calculate the standard deviation for each column in the frame.

Parameters
----------
return: Frame
    The frame will have one row and the same number/names
    of columns as in the current frame. All the columns
    will have `float64` stype. For non-numeric columns
    this function returns NA values.

See Also
--------
- :meth:`.sd1()` -- similar to this method, but operates on a
  single-column frame only, and returns a scalar value instead of
  a Frame.


- :func:`dt.sd()` -- function for calculating the standard deviation
  in a column or an expression; can also be applied per-group.
)";


const char* doc_Frame_shape =
R"(Tuple with ``(nrows, ncols)`` dimensions of the frame.

This property is read-only.

Parameters
----------
return: Tuple[int, int]
    Tuple with two integers: the first is the number of rows, the
    second is the number of columns.

See also
--------
- :attr:`.nrows` -- getter for the number of rows;
- :attr:`.ncols` -- getter for the number of columns.
)";


const char* doc_Frame_skew1 =
R"(skew1(self)
--

Calculate the skewness for a one-column frame and return it as a scalar.

This function is a shortcut for::

    DT.skew()[0, 0]

Parameters
----------
return: None | float
    `None` is returned for non-numeric columns.

except: ValueError
    If called on a Frame that has more or less than one column.

See Also
--------
- :meth:`.skew()` -- similar to this method, but can be applied to
  a Frame with an arbitrary number of columns.
)";


const char* doc_Frame_skew =
R"(skew(self)
--

Calculate the skewness for each column in the frame.

Parameters
----------
return: Frame
    The frame will have one row and the same number/names
    of columns as in the current frame. All the columns
    will have `float64` stype. For non-numeric columns
    this function returns NA values.

See Also
--------
- :meth:`.skew1()` -- similar to this method, but operates on a
  single-column frame only, and returns a scalar value instead of
  a Frame.
)";


const char* doc_Frame_sort =
R"(sort(self, *cols)
--

Sort frame by the specified column(s).


Parameters
----------
cols: List[str | int]
    Names or indices of the columns to sort by. If no columns are
    given, the Frame will be sorted on all columns.

return: Frame
    New Frame sorted by the provided column(s). The current frame
    remains unmodified.
)";


const char* doc_Frame_source =
R"(.. x-version-added:: 0.11

The name of the file where this frame was loaded from.

This is a read-only property that describes the origin of the frame.
When a frame is loaded from a Jay or CSV file, this property will
contain the name of that file. Similarly, if the frame was opened
from a URL or a from a shell command, the source will report the
original URL / the command.

Certain sources may be converted into a Frame only partially,
in such case the ``source`` property will attempt to reflect this
fact. For example, when opening a multi-file zip archive, the
source will contain the name of the file within the archive.
Similarly, when opening an XLS file with several worksheets, the
source property will contain the name of the XLS file, the name of
the worksheet, and possibly even the range of cells that were read.

Parameters
----------
return: str | None
    If the frame was loaded from a file or similar resource, the
    name of that file is returned. If the frame was computed, or its
    data modified, the property will return ``None``.
)";


const char* doc_Frame_stype =
R"(.. x-version-deprecated:: 1.0.0

    This property is deprecated and will be removed in version 1.2.0.
    Please use :attr:`.types` instead.

.. x-version-added:: 0.10.0

The common :class:`dt.stype` for all columns.

This property is well-defined only for frames where all columns have
the same stype.

Parameters
----------
return: stype | None
    For frames where all columns have the same stype, this common
    stype is returned. If a frame has 0 columns, `None` will be
    returned.

except: InvalidOperationError
    This exception will be raised if the columns in the frame have
    different stypes.

See also
--------
- :attr:`.stypes` -- tuple of stypes for all columns.
)";


const char* doc_Frame_stypes =
R"(.. x-version-deprecated:: 1.0.0

    This property is deprecated and will be removed in version 1.2.0.
    Please use :attr:`.types` instead.

The tuple of each column's stypes ("storage types").

Parameters
----------
return: Tuple[stype, ...]
    The length of the tuple is the same as the number of columns in
    the frame.

See also
--------
- :attr:`.stype` -- common stype for all columns
- :attr:`.ltypes` -- tuple of columns' logical types
)";


const char* doc_Frame_sum1 =
R"(sum1(self)
--

Calculate the sum of all values for a one-column column frame and
return it as a scalar.

This function is a shortcut for::

    DT.sum()[0, 0]

Parameters
----------
return: None | float
    `None` is returned for non-numeric columns.

except: ValueError
    If called on a Frame that has more or less than one column.

See Also
--------
- :meth:`.sum()` -- similar to this method, but can be applied to
  a Frame with an arbitrary number of columns.

- :func:`dt.sum()` -- function for calculating the sum of all the values
  in a column or an expression; can also be applied per-group.
)";


const char* doc_Frame_sum =
R"(sum(self)
--

Calculate the sum of all values for each column in the frame.

Parameters
----------
return: Frame
    The resulting frame will have one row and the same
    number/names of columns as in the original frame.
    The column types are :attr:`int64 <dt.Type.int64>` for
    void, boolean and integer columns, :attr:`float32 <dt.Type.float32>`
    for :attr:`float32 <dt.Type.float32>` columns and
    :attr:`float64 <dt.Type.float64>` for :attr:`float64 <dt.Type.float64>`
    columns. For non-numeric columns an NA :attr:`float64 <dt.Type.float64>`
    column is returned.

See Also
--------
- :meth:`.sum1()` -- similar to this method, but operates on a
  single-column frame only, and returns a scalar value instead of
  a Frame.

- :func:`dt.sum()` -- function for calculating the sum of all the values
  in a column or an expression; can also be applied per-group.
)";


const char* doc_Frame_tail =
R"(tail(self, n=10)
--

Return the last `n` rows of the frame.

If the number of rows in the frame is less than `n`, then all rows
are returned.

This is a convenience function and it is equivalent to `DT[-n:, :]`
(except when `n` is 0).

Parameters
----------
n : int
    The maximum number of rows to return, 10 by default. This number
    cannot be negative.

return: Frame
    A frame containing the last up to `n` rows from the original
    frame, and same columns.


Examples
--------

>>> DT = dt.Frame(A=["apples", "bananas", "cherries", "dates",
...                  "eggplants", "figs", "grapes", "kiwi"])
>>> DT.tail(3)
   | A
   | str32
-- + ------
 0 | figs
 1 | grapes
 2 | kiwi
[3 rows x 1 column]


See also
--------
- :meth:`.head` -- return the first `n` rows of the Frame.
)";


const char* doc_Frame_to_arrow =
R"(to_arrow(self)
--

Convert this frame into a ``pyarrow.Table`` object. The ``pyarrow``
module must be installed.

The conversion is multi-threaded and done in C++, but it does
involve creating a copy of the data, except for the cases when the
data was originally imported from Arrow. This is caused by differences
in the data storage formats of datatable and Arrow.

Parameters
----------
return: pyarrow.Table
    A ``Table`` object is always returned, even if the source is a
    single-column datatable Frame.

except: ImportError
    If the `pyarrow` module is not installed.
)";


const char* doc_Frame_to_csv =
R"(to_csv(self, path=None, *, sep =",", quoting="minimal",
--

Write the contents of the Frame into a CSV file.

This method uses multiple threads to serialize the Frame's data. The
number of threads is can be configured using the global option
``dt.options.nthreads``.

The method supports simple writing to file, appending to an existing
file, or creating a python string if no filename was provided.
Optionally, the output could be gzip-compressed.


Parameters
----------
path: str
    Path to the output CSV file that will be created. If the file
    already exists, it will be overwritten. If no path is given,
    then the Frame will be serialized into a string, and that string
    will be returned.

sep: str
    Field separator, must be a single-character string. To be
    able to read the resulting CSV file back to datatable,
    the separator should meet requirements of :func:`dt.fread()`.

quoting: csv.QUOTE_* | "minimal" | "all" | "nonnumeric" | "none"
    `"minimal"` | `csv.QUOTE_MINIMAL`
        Quote the string fields only as necessary, i.e. if the string
        starts or ends with the whitespace, or contains quote
        characters, separator, or any of the C0 control characters
        (including newlines, etc).

    `"all"` | `csv.QUOTE_ALL`
        Fields for all the column types will be quoted.

    `"nonnumeric"` | `csv.QUOTE_NONNUMERIC`
        All string fields will be quoted.

    `"none"` | `csv.QUOTE_NONE`
        None of the fields will be quoted. This option must be used
        at user's own risk: the file produced may not be valid CSV.

    .. note::
        To use `csv.QUOTE_*` constants,
        `csv <https://docs.python.org/3/library/csv.html>`_ module
        must be imported first.

append: bool
    If True, the file given in the `path` parameter will be opened
    for appending (i.e. mode="a"), or created if it doesn't exist.
    If False (default), the file given in the `path` will be
    overwritten if it already exists.

header: bool | "auto"
    This option controls whether or not to write headers into the
    output file. If this option is not given (or equal to ...), then
    the headers will be written unless the option `append` is True
    and the file `path` already exists. Thus, by default the headers
    will be written in all cases except when appending content into
    an existing file.

bom: bool
    If True, then insert the byte-order mark into the output file
    (the option is False by default). Even if the option is True,
    the BOM will not be written when appending data to an existing
    file.

    According to Unicode standard, including BOM into text files is
    "neither required nor recommended". However, some programs (e.g.
    Excel) may not be able to recognize file encoding without this
    mark.

hex: bool
    If True, then all floating-point values will be printed in hex
    format (equivalent to %a format in C `printf`). This format is
    around 3 times faster to write/read compared to usual decimal
    representation, so its use is recommended if you need maximum
    speed.

compression: None | "gzip" | "auto"
    Which compression method to use for the output stream. The default
    is "auto", which tries to infer the compression method from the
    output file's name. The only compression format currently supported
    is "gzip". Compression may not be used when `append` is True.

verbose: bool
    If True, some extra information will be printed to the console,
    which may help to debug the inner workings of the algorithm.

method: "mmap" | "write" | "auto"
    Which method to use for writing to disk. On certain systems 'mmap'
    gives a better performance; on other OSes 'mmap' may not work at
    all.

return: None | str | bytes
    None if `path` is non-empty. This is the most common case: the
    output is written to the file provided.

    String containing the CSV text as if it would have been written
    to a file, if the path is empty or None. If the compression is
    turned on, a bytes object will be returned instead.
)";


const char* doc_Frame_to_dict =
R"(to_dict(self)
--

Convert the frame into a dictionary of lists, by columns.

The order of records in the dictionary will be the
same as the order of columns in the frame.

Parameters
----------
return: Dict[str, List]
    Dictionary with :attr:`.ncols` records. Each record
    represents a single column: the key is the column's name, and the
    value is the list with the column's data.

Examples
--------
>>> DT = dt.Frame(A=[1, 2, 3], B=["aye", "nay", "tain"])
>>> DT.to_dict()
{"A": [1, 2, 3], "B": ["aye", "nay", "tain"]}

See also
--------
- :meth:`.to_list`: convert the frame into a list of lists
- :meth:`.to_tuples`: convert the frame into a list of tuples by rows
)";


const char* doc_Frame_to_jay =
R"(to_jay(self, path=None, method='auto')
--

Save this frame to a binary file on disk, in `.jay` format.

Parameters
----------
path: str | None
    The destination file name. Although not necessary, we recommend
    using extension ".jay" for the file. If the file exists, it will
    be overwritten.
    If this argument is omitted, the file will be created in memory
    instead, and returned as a `bytes` object.

method: 'mmap' | 'write' | 'auto'
    Which method to use for writing the file to disk. The "write"
    method is more portable across different operating systems, but
    may be slower. This parameter has no effect when `path` is
    omitted.

return: None | bytes
    If the `path` parameter is given, this method returns nothing.
    However, if `path` was omitted, the return value is a `bytes`
    object containing encoded frame's data.
)";


const char* doc_Frame_to_list =
R"(to_list(self)
--

Convert the frame into a list of lists, by columns.


Parameters
----------
return: List[List]
    A list of :attr:`.ncols` lists, each inner list
    representing one column of the frame.


Examples
--------
>>> DT = dt.Frame(A=[1, 2, 3], B=["aye", "nay", "tain"])
>>> DT.to_list()
[[1, 2, 3], ["aye", "nay", "tain"]]

>>> dt.Frame(id=range(10)).to_list()
[[0, 1, 2, 3, 4, 5, 6, 7, 8, 9]]
)";


const char* doc_Frame_to_numpy =
R"(to_numpy(self, type=None, column=None, c_contiguous=False)
--

Convert frame into a 2D numpy array, optionally forcing it into the
specified type.

In a limited set of circumstances the returned numpy array will be
created as a data view, avoiding copying the data. This happens if
all of these conditions are met:

  - the frame has only one column, which is not virtual;
  - the column's type is not string;
  - the `type` argument was not used.

In all other cases the returned numpy array will have a copy of the
frame's data. If the frame has multiple columns of different stypes,
then the values will be upcasted into the smallest common stype.

If the frame has any NA values, then the returned numpy array will
be an instance of `numpy.ma.masked_array`.


Parameters
----------
type: Type | <type-like>
    Cast frame into this type before converting it into a numpy
    array. Here "type-like" can be any value that is acceptable to the
    :class:`dt.Type` constructor.

column: int
    Convert a single column instead of the whole frame. This column index
    can be negative, indicating columns counted from the end of the frame.

c_contiguous: bool
    If `False`, the resulting numpy array will have the same memory
    layout as the datatable frame, i.e. column-major or `C_CONTIGUOUS`.
    If `True`, the numpy array will be row-major or `F_CONTIGUOUS`.

return: numpy.ndarray | numpy.ma.core.MaskedArray
    The returned array will be 2-dimensional with the same :attr:`.shape`
    as the original frame. However, if the option `column` was used,
    then the returned array will be 1-dimensional with the length of
    :attr:`.nrows`.

    A masked array is returned if the frame contains NA values but the
    corresponding numpy array does not support NAs.

except: ImportError
    If the ``numpy`` module is not installed.
)";


const char* doc_Frame_to_pandas =
R"(to_pandas(self)
--

Convert this frame into a pandas DataFrame.

If the frame being converted has one or more key columns, those
columns will become the index in the pandas DataFrame.

Parameters
----------
return: pandas.DataFrame
    Pandas dataframe of shape ``(nrows, ncols-nkeys)``.

except: ImportError
    If the `pandas` module is not installed.
)";


const char* doc_Frame_to_tuples =
R"(to_tuples(self)
--

Convert the frame into a list of tuples, by rows.

Parameters
----------
return: List[Tuple]
  Returns a list having :attr:`.nrows` tuples, where
  each tuple has length :attr:`.ncols` and contains data
  from each respective row of the frame.

Examples
--------
>>> DT = dt.Frame(A=[1, 2, 3], B=["aye", "nay", "tain"])
>>> DT.to_tuples()
[(1, "aye"), (2, "nay"), (3, "tain")]
)";


const char* doc_Frame_type =
R"(.. x-version-added:: v1.0.0

The common :class:`dt.Type` for all columns.

This property is well-defined only for frames where all columns have
the same type.

Parameters
----------
return: Type | None
    For frames where all columns have the same type, this common
    type is returned. If a frame has 0 columns, `None` will be
    returned.

except: InvalidOperationError
    This exception will be raised if the columns in the frame have
    different types.

See also
--------
- :attr:`.types` -- list of types for all columns.
)";


const char* doc_Frame_types =
R"(.. x-version-added:: v1.0.0

The list of `Type`s for each column of the frame.

Parameters
----------
return: List[Type]
    The length of the list is the same as the number of columns in the frame.


See also
--------
- :attr:`.type` -- common type for all columns
)";


const char* doc_Frame_view =
R"(view(self, interactive=None, plain=False)
--

.. warning::

    This function is currently not working properly.
    `[#2669] <https://github.com/h2oai/datatable/issues/2669>`_
)";


const char* doc_FExpr =
R"(``FExpr`` is a class that encapsulates computations to be done on a frame.

``FExpr`` objects are rarely constructed directly (though it is possible too),
instead they are more commonly created as inputs/outputs from various
functions in :mod:`datatable`.

Consider the following example::

    math.sin(2 * f.Angle)

Here accessing column "Angle" in namespace ``f`` creates an ``FExpr`` object.
Multiplying this ``FExpr`` by a python scalar ``2`` creates a new ``FExpr``.
And finally, applying the sine function creates yet another ``FExpr``. The
resulting expression can be applied to a frame via the
:meth:`DT[i, j] <dt.Frame.__getitem__>` method, which will compute that expression
using the data of that particular frame.

Thus, an ``FExpr`` is a stored computation, which can later be applied to a
Frame, or to multiple frames.

Because of its delayed nature, an ``FExpr`` checks its correctness at the time
when it is applied to a frame, not sooner. In particular, it is possible for
the same expression to work on one frame, but fail on another. In the
example above, the expression may raise an error if there is no column named
"Angle" in the frame, or if the column exists but has non-numeric type.

Most functions in datatable that accept an ``FExpr`` as an input, return
a new ``FExpr`` as an output, thus creating a tree of ``FExpr``s as the
resulting evaluation graph.

Also, all functions that accept ``FExpr``s as arguments, will also accept
certain other python types as an input, essentially converting them into
``FExpr``s. Hence, we will sometimes say that a function accepts **FExpr-like**
objects as arguments.

All binary operators ``op(x, y)`` listed below work when either ``x``
or ``y``, or both are ``FExpr``s.


Construction
------------

.. list-table::
    :widths: auto
    :class: api-table

    * - :meth:`.__init__(e)`
      - Create an ``FExpr``.

    * - :meth:`.alias()`
      - Assign new names to the columns from the ``FExpr``.

    * - :meth:`.extend()`
      - Append another ``FExpr``.

    * - :meth:`.remove()`
      - Remove columns from the ``FExpr``.


Arithmeritc operators
---------------------

.. list-table::
    :widths: auto
    :class: api-table

    * - :meth:`__add__(x, y) <dt.FExpr.__add__>`
      - Addition ``x + y``.

    * - :meth:`__sub__(x,  y) <dt.FExpr.__sub__>`
      - Subtraction ``x - y``.

    * - :meth:`__mul__(x, y) <dt.FExpr.__mul__>`
      - Multiplication ``x * y``.

    * - :meth:`__truediv__(x, y) <dt.FExpr.__truediv__>`
      - Division ``x / y``.

    * - :meth:`__floordiv__(x, y) <dt.FExpr.__floordiv__>`
      - Integer division ``x // y``.

    * - :meth:`__mod__(x, y) <dt.FExpr.__mod__>`
      - Modulus ``x % y`` (the remainder after integer division).

    * - :meth:`__pow__(x, y) <dt.FExpr.__pow__>`
      - Power ``x ** y``.

    * - :meth:`__pos__(x) <dt.FExpr.__pos__>`
      - Unary plus ``+x``.

    * - :meth:`__neg__(x) <dt.FExpr.__neg__>`
      - Unary minus ``-x``.


Bitwise operators
-----------------

.. list-table::
    :widths: auto
    :class: api-table

    * - :meth:`__and__(x, y) <dt.FExpr.__and__>`
      - Bitwise AND ``x & y``.

    * - :meth:`__or__(x, y) <dt.FExpr.__or__>`
      - Bitwise OR ``x | y``.

    * - :meth:`__xor__(x, y) <dt.FExpr.__xor__>`
      - Bitwise XOR ``x ^ y``.

    * - :meth:`__invert__(x) <dt.FExpr.__invert__>`
      - Bitwise NOT ``~x``.

    * - :meth:`__lshift__(x, y) <dt.FExpr.__lshift__>`
      - Left shift ``x << y``.

    * - :meth:`__rshift__(x, y) <dt.FExpr.__rshift__>`
      - Right shift ``x >> y``.


Relational operators
--------------------

.. list-table::
    :widths: auto
    :class: api-table

    * - :meth:`__eq__(x, y) <dt.FExpr.__eq__>`
      - Equal ``x == y``.

    * - :meth:`__ne__(x, y) <dt.FExpr.__ne__>`
      - Not equal ``x != y``.

    * - :meth:`__lt__(x, y) <dt.FExpr.__lt__>`
      - Less than ``x < y``.

    * - :meth:`__le__(x, y) <dt.FExpr.__le__>`
      - Less than or equal ``x <= y``.

    * - :meth:`__gt__(x, y) <dt.FExpr.__gt__>`
      - Greater than ``x > y``.

    * - :meth:`__ge__(x, y) <dt.FExpr.__ge__>`
      - Greater than or equal ``x >= y``.


Equivalents of base datatable functions
---------------------------------------

.. list-table::
    :widths: auto
    :class: api-table

    * - :meth:`.as_type()`
      - Same as :func:`dt.as_type()`.

    * - :meth:`.categories()`
      - Same as :func:`dt.categories()`.

    * - :meth:`.codes()`
      - Same as :func:`dt.codes()`.

    * - :meth:`.count()`
      - Same as :func:`dt.count()`.

    * - :meth:`.countna()`
      - Same as :func:`dt.countna()`.

    * - :meth:`.cummax()`
      - Same as :func:`dt.cummax()`.

    * - :meth:`.cummin()`
      - Same as :func:`dt.cummin()`.

    * - :meth:`.cumprod()`
      - Same as :func:`dt.cumprod()`.

    * - :meth:`.cumsum()`
      - Same as :func:`dt.cumsum()`.

    * - :meth:`.fillna()`
      - Same as :func:`dt.fillna()`.

    * - :meth:`.first()`
      - Same as :func:`dt.first()`.

    * - :meth:`.last()`
      - Same as :func:`dt.last()`.

    * - :meth:`.max()`
      - Same as :func:`dt.max()`.

    * - :meth:`.mean()`
      - Same as :func:`dt.mean()`.

    * - :meth:`.median()`
      - Same as :func:`dt.median()`.

    * - :meth:`.min()`
      - Same as :func:`dt.min()`.

    * - :meth:`.nunique()`
      - Same as :func:`dt.nunique()`.

    * - :meth:`.prod()`
      - Same as :func:`dt.prod()`.

    * - :meth:`.rowall()`
      - Same as :func:`dt.rowall()`.

    * - :meth:`.rowany()`
      - Same as :func:`dt.rowany()`.

    * - :meth:`.rowcount()`
      - Same as :func:`dt.rowcount()`.

    * - :meth:`.rowfirst()`
      - Same as :func:`dt.rowfirst()`.

    * - :meth:`.rowlast()`
      - Same as :func:`dt.rowlast()`.

    * - :meth:`.rowargmax()`
      - Same as :func:`dt.rowargmax()`.

    * - :meth:`.rowmax()`
      - Same as :func:`dt.rowmax()`.

    * - :meth:`.rowmean()`
      - Same as :func:`dt.rowmean()`.

    * - :meth:`.rowargmin()`
      - Same as :func:`dt.rowargmin()`.

    * - :meth:`.rowmin()`
      - Same as :func:`dt.rowmin()`.

    * - :meth:`.rowsd()`
      - Same as :func:`dt.rowsd()`.

    * - :meth:`.rowsum()`
      - Same as :func:`dt.rowsum()`.

    * - :meth:`.sd()`
      - Same as :func:`dt.sd()`.

    * - :meth:`.shift()`
      - Same as :func:`dt.shift()`.

    * - :meth:`.sum()`
      - Same as :func:`dt.sum()`.

Miscellaneous
-------------

.. list-table::
    :widths: auto
    :class: api-table

    * - :meth:`.__bool__()`
      - Implicitly convert ``FExpr`` into a boolean value.

    * - :meth:`.__getitem__()`
      - Apply slice to a string column.

    * - :meth:`.__repr__()`
      - Used by Python function :ext-func:`repr() <repr>`.

    * - :meth:`.len()`
      - String length.

    * - :meth:`.re_match(pattern)`
      - Check whether the string column matches a pattern.
)";


const char* doc_FExpr_alias =
R"(alias(self, *names)
--

Assign new names to the columns from the current ``FExpr``.


Parameters
----------
names: str | List[str] | Tuple[str]
    New names that should be assigned to the columns from
    the current ``FExpr``.

return: FExpr
    New ``FExpr`` which sets new `names` on the current one.


Examples
--------

Create a frame::

    >>> from datatable import dt, f, by
    >>>
    >>> DT = dt.Frame([[1, 2, 3], ["one", "two", "three"]])
    >>> DT
       |    C0  C1
       | int32  str32
    -- + -----  -----
     0 |     1  one
     1 |     2  two
     2 |     3  three
    [3 rows x 2 columns]


Assign new names when selecting data from the frame::

    >>> DT[:, f[:].alias("numbers", "strings")]
       | numbers  strings
       |   int32  str32
    -- + -------  -------
     0 |       1  one
     1 |       2  two
     2 |       3  three
    [3 rows x 2 columns]


Assign new name for the newly computed column only::

    >>> DT[:, [f[:], (f[0] * f[0]).alias("numbers_squared")]]
       |    C0  C1     numbers_squared
       | int32  str32            int32
    -- + -----  -----  ---------------
     0 |     1  one                  1
     1 |     2  two                  4
     2 |     3  three                9
    [3 rows x 3 columns]


Assign new name for the group by column::

    >>> DT[:, f[1], by(f[0].alias("numbers"))]
       | numbers  C1
       |   int32  str32
    -- + -------  -----
     0 |       1  one
     1 |       2  two
     2 |       3  three
    [3 rows x 2 columns]
)";


const char* doc_FExpr_as_type =
R"(as_type(new_type)
--

.. x-version-added:: 1.1.0

Equivalent to :func:`dt.as_type(cols, new_type)`.
)";


const char* doc_FExpr_categories =
R"(categories()
--

Equivalent to :func:`dt.categories(cols)`.
)";


const char* doc_FExpr_codes =
R"(codes()
--

Equivalent to :func:`dt.codes(cols)`.
)";


const char* doc_FExpr_count =
R"(count()
--

Equivalent to :func:`dt.count(cols)`.
)";


const char* doc_FExpr_countna =
R"(countna()
--

Equivalent to :func:`dt.countna(cols)`.
)";


const char* doc_FExpr_cummax =
R"(cummax()
--

Equivalent to :func:`dt.cummax(cols)`.
)";


const char* doc_FExpr_cummin =
R"(cummin()
--

Equivalent to :func:`dt.cummin(cols)`.
)";


const char* doc_FExpr_cumprod =
R"(cumprod()
--

Equivalent to :func:`dt.cumprod(cols)`.
)";


const char* doc_FExpr_cumsum =
R"(cumsum()
--

Equivalent to :func:`dt.cumsum(cols)`.
)";


const char* doc_FExpr_extend =
R"(extend(self, arg)
--

Append ``FExpr`` `arg` to the current FExpr.

Each ``FExpr`` represents a collection of columns, or a columnset. This
method takes two such columnsets and combines them into a single one,
similar to :func:`cbind() <datatable.cbind>`.


Parameters
----------
arg: FExpr
    The expression to append.

return: FExpr
    New FExpr which is a combination of the current FExpr and `arg`.


See also
--------
- :meth:`remove() <dt.FExpr.remove>` -- remove columns from a columnset.
)";


const char* doc_FExpr_fillna =
R"(fillna(reverse=False)
--

Equivalent to :func:`dt.fillna(cols, reverse=False)`.
)";


const char* doc_FExpr_first =
R"(first()
--

Equivalent to :func:`dt.first(cols)`.
)";


const char* doc_FExpr_last =
R"(last()
--

Equivalent to :func:`dt.last(cols)`.
)";


const char* doc_FExpr_max =
R"(max()
--

Equivalent to :func:`dt.max(cols)`.
)";


const char* doc_FExpr_mean =
R"(mean()
--

Equivalent to :func:`dt.mean(cols)`.
)";


const char* doc_FExpr_median =
R"(median()
--

Equivalent to :func:`dt.median(cols)`.
)";


const char* doc_FExpr_min =
R"(min()
--

Equivalent to :func:`dt.min(cols)`.
)";


const char* doc_FExpr_nunique =
R"(nunique()
--

Equivalent to :func:`dt.nunique(cols)`.
)";


const char* doc_FExpr_prod =
R"(prod()
--

.. x-version-added:: 1.1.0

Equivalent to :func:`dt.prod(cols)`.
)";


const char* doc_FExpr_remove =
R"(remove(self, arg)
--

Remove columns `arg` from the current FExpr.

Each ``FExpr`` represents a collection of columns, or a columnset. Some
of those columns are computed while others are specified "by reference",
for example ``f.A``, ``f[:3]`` or ``f[int]``. This method allows you to
remove by-reference columns from an existing FExpr.


Parameters
----------
arg: FExpr
    The columns to remove. These must be "columns-by-reference", i.e.
    they cannot be computed columns.

return: FExpr
    New FExpr which is a obtained from the current FExpr by removing
    the columns in `arg`.


See also
--------
- :meth:`extend() <dt.FExpr.extend>` -- append a columnset.
)";


const char* doc_FExpr_rowall =
R"(rowall()
--

Equivalent to :func:`dt.rowall(*cols)`.
)";


const char* doc_FExpr_rowany =
R"(rowany()
--

Equivalent to :func:`dt.rowany(*cols)`.
)";


const char* doc_FExpr_rowcount =
R"(rowcount()
--

Equivalent to :func:`dt.rowcount(*cols)`.
)";


const char* doc_FExpr_rowfirst =
R"(rowfirst()
--

Equivalent to :func:`dt.rowfirst(*cols)`.
)";


const char* doc_FExpr_rowlast =
R"(rowlast()
--

Equivalent to :func:`dt.rowlast(*cols)`.
)";


const char* doc_FExpr_rowargmax =
R"(rowargmax()
--

.. x-version-added:: 1.1.0

Equivalent to :func:`dt.rowargmax(*cols)`.
)";


const char* doc_FExpr_rowargmin =
R"(rowargmin()
--

.. x-version-added:: 1.1.0

Equivalent to :func:`dt.rowargmin(*cols)`.
)";


const char* doc_FExpr_rowmax =
R"(rowmax()
--

Equivalent to :func:`dt.rowmax(*cols)`.
)";


const char* doc_FExpr_rowmean =
R"(rowmean()
--

Equivalent to :func:`dt.rowmean(*cols)`.
)";


const char* doc_FExpr_rowmin =
R"(rowmin()
--

Equivalent to :func:`dt.rowmin(*cols)`.
)";


const char* doc_FExpr_rowsd =
R"(rowsd()
--

Equivalent to :func:`dt.rowsd(*cols)`.
)";


const char* doc_FExpr_rowsum =
R"(rowsum()
--

Equivalent to :func:`dt.rowsum(*cols)`.
)";


const char* doc_FExpr_sd =
R"(sd()
--

Equivalent to :func:`dt.sd(cols)`.
)";


const char* doc_FExpr_shift =
R"(shift(n=1)
--

Equivalent to :func:`dt.shift(cols, n=1)`.
)";


const char* doc_FExpr_sum =
R"(sum()
--

Equivalent to :func:`dt.sum(cols)`.
)";


const char* doc_Namespace =
R"(A namespace is an environment that provides lazy access to columns of
a frame when performing computations within
:meth:`DT[i,j,...] <dt.Frame.__getitem__>`.

This class should not be instantiated directly, instead use the
singleton instances :data:`f <datatable.f>` and :data:`g <datatable.g>`
exported from the :mod:`datatable` module.


Special methods
---------------

.. list-table::
    :widths: auto
    :class: api-table

    * - :meth:`.__getattribute__(attr)`
      - Access columns as attributes.

    * - :meth:`.__getitem__(item)`
      - Access columns by their names / indices.
)";


const char* doc_Type =
R"(.. x-version-added:: 1.0.0

Type of data stored in a single column of a Frame.

The type describes both the logical meaning of the data (i.e. an integer,
a floating point number, a string, etc.), as well as storage requirement
of that data (the number of bits per element). Some types may carry
additional properties, such as a timezone  or precision.

.. note::

    This property replaces previous :class:`dt.stype` and :class:`dt.ltype`.


Values
------
The following types are currently available:

- :meth:`dt.Type.arr32(T)`
- :meth:`dt.Type.arr64(T)`
- :attr:`dt.Type.bool8`
- :attr:`dt.Type.date32`
- :meth:`dt.Type.cat8(T)`
- :meth:`dt.Type.cat16(T)`
- :meth:`dt.Type.cat32(T)`
- :attr:`dt.Type.float32`
- :attr:`dt.Type.float64`
- :attr:`dt.Type.int8`
- :attr:`dt.Type.int16`
- :attr:`dt.Type.int32`
- :attr:`dt.Type.int64`
- :attr:`dt.Type.obj64`
- :attr:`dt.Type.str32`
- :attr:`dt.Type.str64`
- :attr:`dt.Type.time64`
- :attr:`dt.Type.void`


Properties
----------
.. list-table::
    :widths: auto
    :class: api-table

    * - :attr:`.is_array`
      - Is this an array type?

    * - :attr:`.is_boolean`
      - Is this a boolean type?

    * - :attr:`.is_compound`
      - Is this a compound type?

    * - :attr:`.is_categorical`
      - Is this a categorical type?

    * - :attr:`.is_float`
      - Is this a float type?

    * - :attr:`.is_integer`
      - Is this an integer type?

    * - :attr:`.is_numeric`
      - Is this a numeric type?

    * - :attr:`.is_object`
      - Is this an object type?

    * - :attr:`.is_string`
      - Is this a string type?

    * - :attr:`.is_temporal`
      - Is this a temporal type?

    * - :attr:`.is_void`
      - Is this the void type?

    * - :attr:`.max`
      - The maximum value for this type

    * - :attr:`.min`
      - The minimum value for this type

    * - :attr:`.name`
      - The name of this type
)";


const char* doc_Type_arr32 =
R"(arr32(T)
--

.. x-version-added:: 1.1.0

Array type with 32-bit offsets. In a column of this type, every element
is a list of values of type `T`.


See Also
--------
:meth:`.arr64(T)` -- another array type, but with 64-bit offsets.
)";


const char* doc_Type_arr64 =
R"(arr64(T)
--

.. x-version-added:: 1.1.0

Array type with 64-bit offsets. In a column of this type, every element
is a list of values of type `T`.


See Also
--------
:meth:`.arr32(T)` -- another array type, but with 32-bit offsets.
)";


const char* doc_Type_cat8 =
R"(cat8(T)
--

.. x-version-added:: 1.1.0

Categorical type with 8-bit codes. In a column of this type, every element
is a value of type `T`.


See also
--------
- :meth:`.cat16(T)` -- categorical type with 16-bit codes.
- :meth:`.cat32(T)` -- categorical type with 32-bit codes.
)";


const char* doc_Type_cat16 =
R"(cat16(T)
--

.. x-version-added:: 1.1.0

Categorical type with 16-bit codes. In a column of this type, every element
is a value of type `T`.


See also
--------
- :meth:`.cat8(T)` -- categorical type with 8-bit codes.
- :meth:`.cat32(T)` -- categorical type with 32-bit codes.
)";


const char* doc_Type_cat32 =
R"(cat32(T)
--

.. x-version-added:: 1.1.0

Categorical type with 32-bit codes. In a column of this type, every element
is a value of type `T`.


See also
--------
- :meth:`.cat8(T)` -- categorical type with 8-bit codes.
- :meth:`.cat16(T)` -- categorical type with 16-bit codes.
)";


const char* doc_Type_is_array =
R"(.. x-version-added:: 1.1.0

Test whether this type belongs to the category of "array" types. This
property returns ``True`` for :meth:`arr32(T) <dt.Type.arr32>` and
:meth:`arr64(T) <dt.Type.arr64>` types only.


Parameters
----------
return: bool


Examples
--------
>>> dt.Type.arr32(int).is_array
True
>>> dt.Type.arr64("float32").is_array
True
>>> dt.Type.void.is_array
False
)";


const char* doc_Type_is_boolean =
R"(.. x-version-added:: 1.1.0

Test whether this type belongs to the category of "boolean" types. This
property returns ``True`` for :attr:`bool8 <dt.Type.bool8>` type only.


Parameters
----------
return: bool


Examples
--------
>>> dt.Type.bool8.is_boolean
True
>>> dt.Type.void.is_boolean
False
>>> dt.Type.int8.is_boolean
False
)";


const char* doc_Type_is_categorical =
R"(.. x-version-added:: 1.1.0

Test whether this type is a categorical one. This
property returns ``True`` for :meth:`cat8(T) <dt.Type.cat8>`,
:meth:`cat16(T) <dt.Type.cat16>` and :meth:`cat32(T) <dt.Type.cat32>` types only.


Parameters
----------
return: bool


Examples
--------
>>> dt.Type.cat8(int).is_categorical
True
>>> dt.Type.cat32(dt.Type.str64).is_categorical
True
>>> dt.Type.void.is_categorical
False
)";


const char* doc_Type_is_compound =
R"(.. x-version-added:: 1.1.0

Test whether this type represents a "compound" type, i.e. whether it may
contain one or more dependent types. Currently, datatable implements only
:meth:`arr32 <dt.Type.arr32>` and :meth:`arr64 <dt.Type.arr64>` compound
types.


Parameters
----------
return: bool


Examples
--------
>>> dt.Type.arr32(dt.int32).is_compound
True
>>> dt.Type.void.is_compound
False
)";


const char* doc_Type_is_float =
R"(.. x-version-added:: 1.1.0

Test whether this type belongs to the category of "float" types. This
property returns ``True`` for :attr:`float32 <dt.Type.float32>` and
:attr:`float64 <dt.Type.float64>` types only.


Parameters
----------
return: bool


Examples
--------
>>> dt.Type.float64.is_float
True
>>> dt.Type.float32.is_float
True
>>> dt.Type.int64.is_float
False
)";


const char* doc_Type_is_integer =
R"(.. x-version-added:: 1.1.0

Test whether this type belongs to the category of "integer" types. This
property returns ``True`` for :attr:`int8 <dt.Type.int8>`,
:attr:`int16 <dt.Type.int16>`, :attr:`int32 <dt.Type.int32>`, and
:attr:`int64 <dt.Type.int64>` types.


Parameters
----------
return: bool


Examples
--------
>>> dt.Type.int8.is_integer
True
>>> dt.Type.int64.is_integer
True
>>> dt.Type.float64.is_integer
False
)";


const char* doc_Type_is_numeric =
R"(.. x-version-added:: 1.1.0

Test whether this type belongs to the category of "numeric" types. All
boolean, integer, float and void types are considered numeric. Therefore,
this method returns ``True`` for types

    - :attr:`bool8 <dt.Type.bool8>`
    - :attr:`int8 <dt.Type.int8>`
    - :attr:`int16 <dt.Type.int16>`
    - :attr:`int32 <dt.Type.int32>`
    - :attr:`int64 <dt.Type.int64>`
    - :attr:`float32 <dt.Type.float32>`
    - :attr:`float64 <dt.Type.float64>`


Parameters
----------
return: bool


Examples
--------
>>> dt.Type.int8.is_numeric
True
>>> dt.Type.float64.is_numeric
True
>>> dt.Type.void.is_numeric
False
>>> dt.Type.bool8.is_numeric
True
>>> dt.Type.str32.is_numeric
False
)";


const char* doc_Type_is_object =
R"(.. x-version-added:: 1.1.0

Test whether this type belongs to the category of "object" types. This
property returns ``True`` for :attr:`obj64 <dt.Type.obj64>` type only.


Parameters
----------
return: bool


Examples
--------
>>> dt.Type.obj64.is_object
True
>>> dt.Type.void.is_object
False
>>> dt.Type.int8.is_object
False
)";


const char* doc_Type_is_string =
R"(.. x-version-added:: 1.1.0

Test whether this type belongs to the category of "string" types. This
property returns ``True`` for :attr:`str32 <dt.Type.str32>` and
:attr:`str64 <dt.Type.str64>` types only.


Parameters
----------
return: bool


Examples
--------
>>> dt.Type.str64.is_string
True
>>> dt.Type.str32.is_string
False
>>> dt.Type.void.is_string
False
)";


const char* doc_Type_is_temporal =
R"(.. x-version-added:: 1.1.0

Test whether this type belongs to the category of time-related types. This
property returns ``True`` for :attr:`date32 <dt.Type.date32>` and
:attr:`time64 <dt.Type.time64>` types only.


Parameters
----------
return: bool


Examples
--------
>>> dt.Type.date32.is_temporal
True
>>> dt.Type.time64.is_temporal
True
>>> dt.Type.str32.is_temporal
False
)";


const char* doc_Type_is_void =
R"(.. x-version-added:: 1.1.0

Returns ``True`` if this type is the :attr:`void <dt.Type.void>` type.


Parameters
----------
return: bool


Examples
--------
>>> dt.Type.void.is_void
True
)";


const char* doc_Type_max =
R"(The largest finite value that this type can represent, if applicable.

Parameters
----------
return: Any
    The type of the returned value corresponds to the Type object: an int
    for integer types, a float for floating-point types, etc. If the type
    has no well-defined max value then `None` is returned.


Examples
--------
>>> dt.Type.int32.max
2147483647
>>> dt.Type.float64.max
1.7976931348623157e+308
>>> dt.Type.date32.max
2146764179


See also
--------
:attr:`.min` -- the smallest value for the type.
)";


const char* doc_Type_min =
R"(The smallest finite value that this type can represent, if applicable.

Parameters
----------
return: Any
    The type of the returned value corresponds to the Type object: an int
    for integer types, a float for floating-point types, etc. If the type
    has no well-defined min value then `None` is returned.


Examples
--------
>>> dt.Type.int8.min
-127
>>> dt.Type.float32.min
-3.4028234663852886e+38
>>> dt.Type.date32.min
-2147483647


See also
--------
:attr:`.max` -- the largest value for the type.
)";


const char* doc_Type_name =
R"(Return the canonical name of this type, as a string.

Examples
--------
>>> dt.Type.int64.name
'int64'
>>> dt.Type(np.bool_).name
'bool8'
)";




}
